/*
 * Copyright (C) 2007-2017 Apple Inc. All rights reserved.
 * Copyright (C) 2015 Canon Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE AND ITS CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE OR ITS CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 

#include "config.h"
#include <wtf/FileSystem.h>
#include <sys/stat.h>
#include <wtf/CryptographicallyRandomNumber.h>
#include <wtf/FileMetadata.h>
#include <wtf/HashMap.h>
#include <wtf/text/CString.h>
#include <wtf/text/StringBuilder.h>
#include <wtf/text/win/WCharStringExtras.h>

//+TELESCOPE_LIB_CHANGES
// 9/3/2020
// Using wrapped system API for File System
extern bool telescopeMkdir(const char* path);
extern bool telescopeGetFileSize(const char* path, long long& outSize);
extern bool telescopeGetFileSize(int fileHandle, long long& outSize);
extern bool telescopeGetFileModificationTime(const char* path, time_t& outSeconds);
extern bool telescopeGetFileCreationTime(const char* path, time_t& outSeconds);
extern bool telescopeFileExists(const char* path);
extern bool telescopeDeleteFile(const char* path);
extern bool telescopeDeleteEmptyDirectory(const char* path);
extern bool telescopeMoveFile(const char* oldPath, const char* newPath);
extern bool telescopeOpenFile(const char* path, int mode, int& outFileHandle);
extern bool telescopeCloseFile(int& handle);
extern bool telescopeFlushFileBuffers(int& handle);
extern long long telescopeSeekFile(int handle, long long offset, int origin);
extern bool telescopeTruncateFile(int fileHandle, long long offset);
extern int telescopeWriteToFile(int handle, const char* data, int length);
extern int telescopeReadFromFile(int handle, char* outData, int length);

typedef void (*DirectoryFileCB)(const void* const pBackParameters, const char* pFileName);
extern void telescopeWalkDirectoryFiles(int handle, const void* const pBackParameters, DirectoryFileCB pFun);
//-TELESCOPE_LIB_CHANGES

namespace WTF {

namespace FileSystemImpl {

bool getFileSize(const String& path, long long& size)
{
	//+TELESCOPE_LIB_CHANGES
	// 9/3/2020
	// Using wrapped system API for File System
	return telescopeGetFileSize(path.ascii().data(), size);
	//-TELESCOPE_LIB_CHANGES
}

bool getFileSize(PlatformFileHandle fileHandle, long long& size)
{
	//+TELESCOPE_LIB_CHANGES
	// 9/3/2020
	// Using wrapped system API for File System
	return telescopeGetFileSize(fileHandle, size);
	//-TELESCOPE_LIB_CHANGES
}

Optional<WallTime> getFileModificationTime(const String& path)
{
	//+TELESCOPE_LIB_CHANGES
	// 9/3/2020
	// Using wrapped system API for File System
	time_t seconds = 0;
	if (telescopeGetFileModificationTime(path.ascii().data(), seconds))
    {
		return WallTime::fromRawSeconds(seconds);
    }

	return WTF::nullopt;
	//-TELESCOPE_LIB_CHANGES
}

Optional<WallTime> getFileCreationTime(const String& path)
{
	//+TELESCOPE_LIB_CHANGES
	// 9/3/2020
	// Using wrapped system API for File System
	time_t seconds = 0;
	if (telescopeGetFileCreationTime(path.ascii().data(), seconds))
	{
		return WallTime::fromRawSeconds(seconds);
	}

	return WTF::nullopt;
}

static String getFinalPathName(const String& path)
{
	assert( 0 );
	return String();
}

Optional<FileMetadata> fileMetadata(const String& path)
{
	//+TELESCOPE_LIB_CHANGES
	// 9/3/2020
	// Using wrapped system API for File System
	auto modificationTime = getFileModificationTime(path);
	if (WTF::nullopt == modificationTime)
	{
		return WTF::nullopt;
	}

	long long fileLength = 0;
	if (!getFileSize(path, fileLength))
	{
		return WTF::nullopt;
	}

	return FileMetadata{
		modificationTime->approximateWallTime(),
		fileLength,
		false,
		FileMetadata::Type::File
	};
	//-TELESCOPE_LIB_CHANGES
}

Optional<FileMetadata> fileMetadataFollowingSymlinks(const String& path)
{
	assert( 0 );
	return WTF::nullopt;
}

bool createSymbolicLink(const String& targetPath, const String& symbolicLinkPath)
{
	assert( 0 );
	return false;
}

bool fileExists(const String& path)
{
	//+TELESCOPE_LIB_CHANGES
	// 9/3/2020
	// Using wrapped system API for File System
	return telescopeFileExists(path.ascii().data());
	//-TELESCOPE_LIB_CHANGES
}

bool deleteFile(const String& path)
{
	//+TELESCOPE_LIB_CHANGES
	// 9/3/2020
	// Using wrapped system API for File System
	return telescopeDeleteFile(path.ascii().data());
	//-TELESCOPE_LIB_CHANGES
}

bool deleteEmptyDirectory(const String& path)
{
	//+TELESCOPE_LIB_CHANGES
	// 9/3/2020
	// Using wrapped system API for File System
	return telescopeDeleteEmptyDirectory(path.ascii().data());
	//-TELESCOPE_LIB_CHANGES
}

bool moveFile(const String& oldPath, const String& newPath)
{
	//+TELESCOPE_LIB_CHANGES
	// 9/3/2020
	// Using wrapped system API for File System
	return telescopeMoveFile(oldPath.ascii().data(), newPath.ascii().data());
	//-TELESCOPE_LIB_CHANGES
}

String pathByAppendingComponent(const String& path, const String& component)
{
    assert( 0 );
#if OS(SONY_PLATFORM)
    if (path.endsWith('\\') || path.endsWith('/'))
    {
        return path + component;
    }
    else
    {
        return path + '/' + component;
    }
#endif
}

String pathByAppendingComponents(StringView path, const Vector<StringView>& components)
{
    assert( 0 );

    String result = path.toString();
    for (auto& component : components)
        result = pathByAppendingComponent(result, component.toString());
    return result;
}

#if !USE(CF)

CString fileSystemRepresentation(const String& path)
{
	return path.utf8();
}

#endif // !USE(CF)

bool DirectoryExists(const wchar_t* path)
{
	assert( 0 );
	return false;
}

bool makeAllDirectories(const String& path)
{
	return telescopeMkdir(path.ascii().data());
}

String homeDirectoryPath()
{
	assert(0);
    return "";
}

String pathGetFileName(const String& path)
{
	//assert(0);
    int lastSlashPos = path.reverseFind('\\');
    int lastBackSlashPath = path.reverseFind('/');
    return path.substring(std::max(lastSlashPos, lastBackSlashPath) + 1);
}

String directoryName(const String& path)
{
	//assert(0);
    String name = path.left(path.length() - pathGetFileName(path).length());
    if (name.characterStartingAt(name.length() - 1) == '\\') {
        // Remove any trailing "\".
        name.truncate(name.length() - 1);
    }
    return name;
}

#if 0
static String bundleName()
{
	assert(0);
    static const NeverDestroyed<String> name = [] {
        String name { "WebKit"_s };

#if USE(CF)
        if (CFBundleRef bundle = CFBundleGetMainBundle()) {
            if (CFTypeRef bundleExecutable = CFBundleGetValueForInfoDictionaryKey(bundle, kCFBundleExecutableKey)) {
                if (CFGetTypeID(bundleExecutable) == CFStringGetTypeID())
                    name = reinterpret_cast<CFStringRef>(bundleExecutable);
            }
        }
#endif

        return name;
    }();

    return name;
}

static String generateTemporaryPath(const Function<bool(const String&)>& action)
{
	assert( 0 );
	return "";
}

#endif

String openTemporaryFile(const String&, PlatformFileHandle& handle)
{
	assert( 0 );
	return "";
}

PlatformFileHandle openFile(const String& path, FileOpenMode mode)
{
	//+TELESCOPE_LIB_CHANGES
	// 9/3/2020
	// Using wrapped system API for File System
	PlatformFileHandle fileHandle;
	if (telescopeOpenFile(path.ascii().data(), static_cast<int>(mode), fileHandle))
	{
		return fileHandle;
	}

	return invalidPlatformFileHandle;
	//-TELESCOPE_LIB_CHANGES
}

bool flushFileBuffers(PlatformFileHandle& handle)
{
	//+TELESCOPE_LIB_CHANGES
	// 7/12/2021
	// Using wrapped system API for File System
	return telescopeFlushFileBuffers(handle);
	//-TELESCOPE_LIB_CHANGES
}

void closeFile(PlatformFileHandle& handle)
{
	//+TELESCOPE_LIB_CHANGES
	// 9/3/2020
	// Using wrapped system API for File System
	telescopeCloseFile(handle);
	//-TELESCOPE_LIB_CHANGES
}

long long seekFile(PlatformFileHandle handle, long long offset, FileSeekOrigin origin)
{
	//+TELESCOPE_LIB_CHANGES
	// 9/3/2020
	// Using wrapped system API for File System
	return telescopeSeekFile(handle, offset, static_cast<int>(origin));
	//-TELESCOPE_LIB_CHANGES
}

bool truncateFile(PlatformFileHandle handle, long long offset)
{
	//+TELESCOPE_LIB_CHANGES
	// 9/3/2020
	// Using wrapped system API for File System
	return telescopeTruncateFile(handle, offset);
	//-TELESCOPE_LIB_CHANGES
}

int writeToFile(PlatformFileHandle handle, const char* data, int length)
{
	//+TELESCOPE_LIB_CHANGES
	// 9/3/2020
	// Using wrapped system API for File System
	return telescopeWriteToFile(handle, data, length);
	//-TELESCOPE_LIB_CHANGES
}

int readFromFile(PlatformFileHandle handle, char* data, int length)
{
	//+TELESCOPE_LIB_CHANGES
	// 9/3/2020
	// Using wrapped system API for File System
	return telescopeReadFromFile(handle, data, length);
	//-TELESCOPE_LIB_CHANGES
}

bool hardLink(const String& source, const String& destination)
{
	assert( 0 );
	return false;
}

bool hardLinkOrCopyFile(const String& source, const String& destination)
{
	assert( 0 );
	return false;
}

Vector<String> listDirectory(const String& directory, const String& filter)
{
	Vector<String> entries;
	auto handle = openFile(directory, FileOpenMode::Read);
	if (!isHandleValid(handle))
	{
		return entries;
	}

	struct addonParameters
	{
		const String* pDirectory;
		Vector<String>* pFileEntries;
	};

	addonParameters cbParameters{ &directory, &entries };

	telescopeWalkDirectoryFiles(handle, (void*)(&cbParameters), [](const void* const pBackParameters, const char* pFileName)
		{
			const addonParameters* pAddonCBParameters = reinterpret_cast<const addonParameters*>(pBackParameters);
			pAddonCBParameters->pFileEntries->append(*(pAddonCBParameters->pDirectory) + pFileName);
		});

	closeFile(handle);
    return entries;
}

bool getVolumeFreeSpace(const String&, uint64_t&)
{
	assert(0);
    return false;
}

Optional<int32_t> getFileDeviceId(const CString& fsFile)
{
	assert( 0 );
	return WTF::nullopt;
}

String realPath(const String& filePath)
{
	assert(0);
    return getFinalPathName(filePath);
}

bool unmapViewOfFile(void* buffer, size_t)
{
	assert( 0 );
	return false;
}

MappedFileData::MappedFileData(const String& filePath, MappedFileMode mode, bool& success)
{
	assert( 0 );
}

bool MappedFileData::mapFileHandle(PlatformFileHandle handle, MappedFileMode)
{
    assert(0);
    return true;
}

} // namespace FileSystemImpl
} // namespace WTF
