/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <wtf/Threading.h>
#include <errno.h>
#include <wtf/HashMap.h>
#include <wtf/Lock.h>
#include <wtf/MainThread.h>
#include <wtf/MathExtras.h>
#include <wtf/NeverDestroyed.h>
#include <wtf/ThreadingPrimitives.h>

#include <mutex>
#include <thread>
#include <chrono>

namespace WTF {

	extern thread_local Thread::ThreadHolder s_threadHolder;

	Thread::ThreadHolder::~ThreadHolder()
	{
		if (thread)
		{
			thread->didExit();
			thread = nullptr;
		}
	}

	Thread::~Thread()
	{
	}

	void Thread::initializeCurrentThreadEvenIfNonWTFCreated()
	{
	}

	void Thread::initializeCurrentThreadInternal(const char* szThreadName)
	{
	}

	void Thread::initializePlatformThreading()
	{
	}

	Thread* Thread::currentMayBeNull()
	{
		return s_threadHolder.thread.get();
	}

	Thread& Thread::initializeTLS(Ref<Thread>&& thread)
	{
		assert(s_threadHolder.thread == nullptr);
		s_threadHolder.thread = WTFMove(thread);
		return *s_threadHolder.thread;
	}

	void Thread::kill()
	{
		scePthreadCancel(m_handle.native_handle());
	}

	static unsigned wtfThreadEntryPoint(void* data)
	{
		Thread::entryPoint(reinterpret_cast<Thread::NewThreadContext*>(data));
		return 0;
	}

	void establishPlatformSpecificHandle(PlatformThreadHandle)
	{
	}

	bool Thread::establishHandle(NewThreadContext* context)
	{
		std::thread threadObj = std::thread(wtfThreadEntryPoint, context);
		m_id = threadObj.get_id();
		m_handle = std::move(threadObj);
		return true;
	}

	void Thread::changePriority(int delta)
	{
		// Unfortunately, std::thread doesn't have suspend or similar stuff.
		// Fortunately, we don't open JIT for sony platform which is the only place use this function.
		ASSERT(false);
	}

	int Thread::waitForCompletion()
	{
		auto lock = holdLock(m_joinMutex);
		if (joinableState() == Joinable)
		{
			ASSERT(m_handle.joinable());
			if (m_handle.joinable())
			{
				m_handle.join();
			}
			didJoin();
		}
		return 0;
	}

	void Thread::detach()
	{
		// We don't allow to detach thread, as we want make sure all threads are terminated before unloading dynamic library.
		ASSERT(false);
	}

	auto Thread::suspend() -> Expected<void, PlatformSuspendError>
	{
		// Unfortunately, std::thread doesn't have suspend or similar stuff.
		// Fortunately, this function seems never be used.
		ASSERT(false);
		return Expected<void, int>();
	}

	void Thread::resume()
	{
		// Unfortunately, std::thread doesn't have resume or similar stuff.
		// Fortunately, this function seems never be used.
		ASSERT(false);
	}

	size_t Thread::getRegisters(PlatformRegisters& registers)
	{
		// Unfortunately, This function is called acommpany with suspend and resume, but both of them are unavailable
		// Fortunately, this function seems never be used.
		ASSERT(false);
		return 0;
	}

	Thread& Thread::initializeCurrentTLS()
	{
		Ref<Thread> thread = adoptRef(*new Thread());

		thread->initializeInThread();
		initializeCurrentThreadEvenIfNonWTFCreated();

		return initializeTLS(WTFMove(thread));
	}

	void Thread::yield()
	{
		std::this_thread::yield();
	}

	Mutex::~Mutex()
	{
	}

	void Mutex::lock()
	{
		m_mutex.lock();
	}

	bool Mutex::tryLock()
	{
		return m_mutex.try_lock();
	}

	void Mutex::unlock()
	{
		m_mutex.unlock();
	}

	ThreadCondition::~ThreadCondition()
	{
	}

	void ThreadCondition::wait(Mutex& mutex)
	{
		std::unique_lock<std::mutex> lk(mutex.impl(), std::defer_lock);
		m_condition.wait(lk);
	}

	bool ThreadCondition::timedWait(Mutex& mutex, WallTime absoluteTime)
	{
		if (absoluteTime < WallTime::now())
		{
			return false;
		}

		if (absoluteTime > WallTime::fromRawSeconds(INT_MAX))
		{
			wait(mutex);
			return true;
		}

		Seconds durationSeconds = absoluteTime - WallTime::now();
		std::chrono::duration<double> duration(durationSeconds.seconds());
		std::unique_lock<std::mutex> lock(mutex.impl(), std::defer_lock);
		return m_condition.wait_for(lock, duration) == std::cv_status::no_timeout;
	}

	void ThreadCondition::signal()
	{
		m_condition.notify_one();
	}

	void ThreadCondition::broadcast()
	{
		m_condition.notify_all();
	}
}
