/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "SystemAPIReplacement.h"
#include <assert.h>

namespace Telescope
{
static SystemAPIWrapperBase *systemAPIWrapper = nullptr;

void InstallSystemAPIWrapper( SystemAPIWrapperBase *systemAPIWrapperImpl )
{
	systemAPIWrapper = systemAPIWrapperImpl;
}
} // namespace Telescope

void telThreadYield()
{
	Telescope::systemAPIWrapper->ThreadYield();
}


void telUsleep( unsigned int microseconds )
{
	Telescope::systemAPIWrapper->SleepMicroseconds( microseconds );
}


uint16_t telNetHtons( uint16_t host16 )
{
	return Telescope::systemAPIWrapper->NetHtons( host16 );
}


uint32_t telNetHtonl( uint32_t host32 )
{
	return Telescope::systemAPIWrapper->NetHtonl( host32 );
}


uint32_t telNetNtohl( uint32_t net32 )
{
	return Telescope::systemAPIWrapper->NetNtohl( net32 );
}

int TEL_NET_SO_NBIO = 0x1200;

// Only PS requires these hidden APIs
#ifdef IS_PS
bool telescopeMkdir( const char *path )
{
	return Telescope::systemAPIWrapper->mkdir( path );
}


bool telescopeGetFileSize( const char *path, long long &outSize )
{
	return Telescope::systemAPIWrapper->getFileSize( path, outSize );
}


bool telescopeGetFileSize( Telescope::SystemAPIWrapperBase::PlatformFileHandle fileHandle, long long &outSize )
{
	return Telescope::systemAPIWrapper->getFileSize( fileHandle, outSize );
}


bool telescopeGetFileModificationTime( const char *path, time_t &outSeconds )
{
	return Telescope::systemAPIWrapper->getFileModificationTime( path, outSeconds );
}


bool telescopeGetFileCreationTime( const char *path, time_t &outSeconds )
{
	return Telescope::systemAPIWrapper->getFileCreationTime( path, outSeconds );
}


bool telescopeFileExists( const char *path )
{
	return Telescope::systemAPIWrapper->fileExists( path );
}


bool telescopeDeleteFile( const char *path )
{
	return Telescope::systemAPIWrapper->deleteFile( path );
}


bool telescopeDeleteEmptyDirectory( const char *path )
{
	return Telescope::systemAPIWrapper->deleteEmptyDirectory( path );
}


bool telescopeMoveFile( const char *oldPath, const char *newPath )
{
	return Telescope::systemAPIWrapper->moveFile( oldPath, newPath );
}


bool telescopeOpenFile( const char *path, int mode, Telescope::SystemAPIWrapperBase::PlatformFileHandle &outFileHandle )
{
	auto fileOperatorMode = static_cast<Telescope::SystemAPIWrapperBase::TelescopeFileOpenMode>( mode );
	assert( fileOperatorMode == Telescope::SystemAPIWrapperBase::TelescopeFileOpenMode::Read || fileOperatorMode == Telescope::SystemAPIWrapperBase::TelescopeFileOpenMode::Write );
	return Telescope::systemAPIWrapper->openFile( path, fileOperatorMode, outFileHandle );
}


bool telescopeFlushFileBuffers( Telescope::SystemAPIWrapperBase::PlatformFileHandle &handle )
{
	return Telescope::systemAPIWrapper->flushFileBuffers( handle );
}


bool telescopeCloseFile( Telescope::SystemAPIWrapperBase::PlatformFileHandle &handle )
{
	return Telescope::systemAPIWrapper->closeFile( handle );
}


long long telescopeSeekFile( Telescope::SystemAPIWrapperBase::PlatformFileHandle handle, long long offset, int origin )
{
	auto seekOrigin = static_cast<Telescope::SystemAPIWrapperBase::TelescopeFileSeekOrigin>( origin );
	assert( seekOrigin == Telescope::SystemAPIWrapperBase::TelescopeFileSeekOrigin::Beginning ||
		seekOrigin == Telescope::SystemAPIWrapperBase::TelescopeFileSeekOrigin::Current ||
		seekOrigin == Telescope::SystemAPIWrapperBase::TelescopeFileSeekOrigin::End );

	return Telescope::systemAPIWrapper->seekFile( handle, offset, seekOrigin );
}


bool telescopeTruncateFile( Telescope::SystemAPIWrapperBase::PlatformFileHandle fileHandle, long long offset )
{
	return Telescope::systemAPIWrapper->truncateFile( fileHandle, offset );
}


int telescopeWriteToFile( Telescope::SystemAPIWrapperBase::PlatformFileHandle handle, const char *data, int length )
{
	return Telescope::systemAPIWrapper->writeToFile( handle, data, length );
}


int telescopeReadFromFile( Telescope::SystemAPIWrapperBase::PlatformFileHandle handle, char *outData, int length )
{
	return Telescope::systemAPIWrapper->readFromFile( handle, outData, length );
}


void telescopeWalkDirectoryFiles( Telescope::SystemAPIWrapperBase::PlatformFileHandle handle, const void *const pBackParameters, Telescope::SystemAPIWrapperBase::DirectoryFileCB pFun )
{
	return Telescope::systemAPIWrapper->walkDirectoryFiles( handle, pBackParameters, pFun );
}


bool telescopeRandomNumber( void *buf, size_t size )
{
	return Telescope::systemAPIWrapper->getRandomNumber( buf, size );
}
#endif // #ifdef IS_PS