/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "config.h"
#include "TelescopeJSGlobal.h"
#include "TelescopeInstance.h"
#include "TelescopeRuntime.h"
#include "TelescopeJavascriptValue.h"
#include "TelescopePropertyIterator.h"
#include "TelescopeJavascriptValue.h"
#include "TelescopeJSBoundObject.h"


namespace JSC
{
namespace Bindings
{
Ref<TelescopeInstance> TelescopeInstance::Create( Telescope::IJSBoundObject *pObj, RefPtr<RootObject> pRootObj )
{
	return adoptRef( *new TelescopeInstance( pObj, pRootObj ) );
}


TelescopeInstance::TelescopeInstance( Telescope::IJSBoundObject *pObj, RefPtr<RootObject> pRootObj )
	: Instance( RefPtr<RootObject>( pRootObj ) )
{
	m_pBoundObject = pObj;
	m_pClass = nullptr;
}


TelescopeInstance::~TelescopeInstance()
{
	m_pBoundObject = nullptr;
	m_pClass = nullptr;
}


Class *TelescopeInstance::getClass() const
{
	if ( !m_pClass )
	{
		m_pClass = TelescopeClass::ClassForInstance( this );
	}

	return m_pClass;
}


JSValue TelescopeInstance::getMethod( JSGlobalObject *pGlobalObject, PropertyName oPropertyName )
{
	return TelescopeRuntimeMethod::create( pGlobalObject, oPropertyName.publicName(), getClass()->methodNamed( oPropertyName, this ) );
}


JSValue TelescopeInstance::invokeMethodInternal( JSGlobalObject *pGlobalObject, const char *pMethodName )
{
	CallFrame *pFrame = pGlobalObject->vm().topCallFrame;
	size_t stCnt = pFrame->argumentCount();
	WTF::Vector<Telescope::TelescopeJSValue> vArgs( stCnt );
	for ( size_t i = 0; i < stCnt; ++i )
	{
		vArgs[i].SetGlobalObject( pGlobalObject );
		JS2Telescope( pGlobalObject, pFrame->uncheckedArgument( i ), &vArgs[i] );
	}

	JSValue oUndefined( jsUndefined() );
	Telescope::TelescopeJSValue oRlt( &oUndefined, pGlobalObject );

	{
		JSLock::DropAllLocks oDropAllLocks( pGlobalObject );
		if ( !( pMethodName ? m_pBoundObject->invokeMethod( pMethodName, vArgs.data(), stCnt, &oRlt ) : m_pBoundObject->invoke( vArgs.data(), stCnt, &oRlt ) ) )
			ASSERT_RETURN_UNDEFINED
	}

	return Telescope2JS( pGlobalObject, oRlt );
}


Telescope::IJSBoundObject *TelescopeInstance::getBoundObj() const
{
	return m_pBoundObject;
}


JSValue TelescopeInstance::invokeMethod( JSGlobalObject *pGlobalObject, CallFrame *pCallFrame, RuntimeMethod *pRuntimeMethod )
{
	if ( !asObject( pRuntimeMethod )->inherits( pGlobalObject->vm(), TelescopeRuntimeMethod::info() ) )
		ASSERT_RETURN_UNDEFINED

	if ( TelescopeMethod *pMethod = static_cast<TelescopeMethod *>( pRuntimeMethod->method() ) )
	{
		return invokeMethodInternal( pGlobalObject, pMethod->TelescopeName() );
	}

	return jsUndefined();
}


bool TelescopeInstance::supportsInvokeDefaultMethod() const
{
	return m_pBoundObject->isFunction();
}


JSValue TelescopeInstance::invokeDefaultMethod( JSGlobalObject *pGlobalObject, CallFrame *pCallFrame )
{
	if ( m_pBoundObject->isFunction() )
	{
		return invokeMethodInternal( pGlobalObject, nullptr );
	}

	return jsUndefined();
}


JSValue TelescopeInstance::defaultValue( JSGlobalObject *pGlobalObject, PreferredPrimitiveType eHint ) const
{
	if ( PreferNumber == eHint )
	{
		return jsNumber( 0 );
	}

	return valueOf( pGlobalObject );
}


JSValue TelescopeInstance::valueOf( JSGlobalObject *pGlobalObject ) const
{
	return JSC::jsString( pGlobalObject->vm(), String( m_pBoundObject->stringValue() ) );
}


void TelescopeInstance::getPropertyNames( JSGlobalObject *pGlobalObject, PropertyNameArray &rNameArr )
{
	JSLock::DropAllLocks oDropAllLocks( pGlobalObject );
	Telescope::JSBoundObjectPropertyIterator *pItr = m_pBoundObject->First();
	while ( pItr )
	{
		rNameArr.add( JSC::Identifier::fromString( pGlobalObject->vm(), pItr->GetKey() ) );
		pItr = m_pBoundObject->GetNext();
	}
}
} // namespace Bindings
} // namespace JSC
