/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "config.h"
#include "TelescopeRuntime.h"
#include "TelescopeInstance.h"
#include "TelescopeJavascriptValue.h"
#include "TelescopeJSBoundObject.h"
#include "JavaScriptCore/JSCJSValueInlines.h"

namespace JSC
{
namespace Bindings
{
//////////////////////////////////////////////////////////////////////////
//	TelescopeClass
typedef HashMap<const Instance *, Class *> InstanceClassMap;
static InstanceClassMap *g_pInstClassMap = nullptr;


TelescopeClass::TelescopeClass()
{
}


TelescopeClass *TelescopeClass::ClassForInstance( const Instance *pInst )
{
	if ( !g_pInstClassMap )
	{
		g_pInstClassMap = new InstanceClassMap;
	}

	Class *pClass = g_pInstClassMap->get( pInst );
	if ( !pClass )
	{
		pClass = new TelescopeClass();
		g_pInstClassMap->set( pInst, pClass );
	}

	return static_cast<TelescopeClass *>( pClass );
}


void TelescopeClass::Finalize()
{
	if ( !g_pInstClassMap )
	{
		return;
	}

	InstanceClassMap::iterator iStart = g_pInstClassMap->begin();
	InstanceClassMap::iterator iEnd = g_pInstClassMap->end();
	for ( InstanceClassMap::iterator iObj = iStart; iObj != iEnd; ++iObj )
		delete iObj->value;

	g_pInstClassMap->clear();
	delete g_pInstClassMap;
	g_pInstClassMap = nullptr;
}


TelescopeClass::~TelescopeClass()
{
	m_mMethods.clear();
	m_mFields.clear();
}


Method *TelescopeClass::methodNamed( PropertyName oPropertyName, Instance *pInst ) const
{
	String sName( oPropertyName.publicName() );
	if ( sName.isNull() )
	{
		return nullptr;
	}

	if ( Method *pMethod = m_mMethods.get( sName.impl() ) )
	{
		return pMethod;
	}

	CString sAnsiiName = sName.ascii();
	const char *pName = sAnsiiName.data();
	Telescope::IJSBoundObject *pObj = static_cast<const TelescopeInstance *>( pInst )->getBoundObj();
	if ( !pObj->hasMethod( pName ) )
	{
		return nullptr;
	}

	auto pUniqueMethod = std::make_unique<TelescopeMethod>( pName );
	TelescopeMethod *pRet = pUniqueMethod.get();
	m_mMethods.set( sName.impl(), std::move( pUniqueMethod ) );

	return pRet;
}


Field *TelescopeClass::fieldNamed( PropertyName oPropertyName, Instance *pInst ) const
{
	String sName( oPropertyName.publicName() );
	if ( sName.isNull() )
	{
		return nullptr;
	}

	if ( Field *pField = m_mFields.get( sName.impl() ) )
	{
		return pField;
	}

	if ( !m_mMethods.get( sName.impl() ) )
	{
		return nullptr;
	}

	CString sAnsiiName = sName.ascii();
	if ( !( static_cast<const TelescopeInstance *>( pInst )->getBoundObj()->hasProperty( sAnsiiName.data() ) ) )
	{
		return nullptr;
	}

	auto pUniqueMethod = std::make_unique<TelescopeField>( sAnsiiName.data() );
	TelescopeField *pRet = pUniqueMethod.get();
	m_mFields.set( sName.impl(), std::move( pUniqueMethod ) );

	return pRet;
}


//////////////////////////////////////////////////////////////////////////
//	TelescopeField
TelescopeField::TelescopeField( const char *pID )
	: m_sIdent( pID )
{
}


const char *TelescopeField::GetName()
{
	return (const char *)( m_sIdent.characters8() );
}


JSValue TelescopeField::valueFromInstance( JSGlobalObject *pGlobalObject, const Instance *pInst ) const
{
	JSValue oUndefined = jsUndefined();
	Telescope::TelescopeJSValue oRlt( &oUndefined, pGlobalObject );

	{
		JSLock::DropAllLocks oDropAllLocks( pGlobalObject );
		static_cast<const TelescopeInstance *>( pInst )->getBoundObj()->getProperty(
			(const char *)( m_sIdent.characters8() ), &oRlt );
	}

	return Telescope2JS( pGlobalObject, oRlt );
}


bool TelescopeField::setValueToInstance( JSGlobalObject *pGlobalObject, const Instance *pInst, JSValue oJSVal ) const
{
	JSValue oUndefined = jsUndefined();
	Telescope::TelescopeJSValue oTelJSVal( &oUndefined, pGlobalObject );
	JS2Telescope( pGlobalObject, oJSVal, &oTelJSVal );

	{
		JSLock::DropAllLocks oDropAllLocks( pGlobalObject );
		static_cast<const TelescopeInstance *>( pInst )->getBoundObj()->setProperty(
			(const char *)( m_sIdent.characters8() ), oTelJSVal );
	}

	return true;
}


//////////////////////////////////////////////////////////////////////////
//	TelescopeMethod
TelescopeMethod::TelescopeMethod( const char *pID )
	: m_sIdent( pID )
{
}


int TelescopeMethod::numParameters() const
{
	assert( 0 );
	return 0;
}


const char *TelescopeMethod::TelescopeName()
{
	return (const char *)( m_sIdent.characters8() );
}


//////////////////////////////////////////////////////////////////////////
//	TelescopeRuntimeMethod
TelescopeRuntimeMethod *TelescopeRuntimeMethod::create( JSGlobalObject *pGlobalObject, const String &rName, Bindings::Method *pMethod )
{
	TelescopeRuntimeMethod *pRuntimeMethod = new ( NotNull,
		allocateCell<TelescopeRuntimeMethod>( *( pGlobalObject->heap() ) ) ) TelescopeRuntimeMethod( pGlobalObject,
		WebCore::deprecatedGetDOMStructure<TelescopeRuntimeMethod>( pGlobalObject ),
		pMethod );
	pRuntimeMethod->finishCreation( pGlobalObject->vm(), rName );

	return pRuntimeMethod;
}


Structure *TelescopeRuntimeMethod::createStructure( VM &rVM, JSGlobalObject *pGlobalObject, JSValue oVal )
{
	return Structure::create( rVM, pGlobalObject, oVal, TypeInfo( ObjectType, StructureFlags ), info() );
}


TelescopeRuntimeMethod::TelescopeRuntimeMethod( JSGlobalObject *pGlobalObject, Structure *pStructure, Bindings::Method *pMethod )
	: RuntimeMethod( pGlobalObject->vm(), pStructure, pMethod )
{
}


void TelescopeRuntimeMethod::finishCreation( VM &rVM, const String &rName )
{
	RuntimeMethod::finishCreation( rVM, rName );
	assert( inherits( rVM, info() ) );
}

const ClassInfo TelescopeRuntimeMethod::s_info =
{
	"TelescopeRuntimeMethod",
	&RuntimeMethod::s_info,
	0,
	nullptr,
	CREATE_METHOD_TABLE( TelescopeRuntimeMethod )
};
} // namespace Bindings
} // namespace JSC
