/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "ClientWrappers/MemoryAllocatorBase.h"
#include "wtf/FastMalloc.h"
#include <cassert>

#ifndef BUILD_TELESCOPE_DLL
#error "The global new/delete override must in dynamic library. Otherwise, it does overides host engine new/delete"
#endif // #ifndef BUILD_TELESCOPE_DLL

TELESCOPE_DEFINE_MEMORY_TAG( GLOBALNEW_MEMORY_TAG, "global new" );

void *operator new( size_t size )
{
	// http://www.cplusplus.com/reference/new/operator%20new/
	// If this argument is zero, the function still returns a distinct non-null pointer on success
	// (although dereferencing this pointer leads to undefined behavior).
	if ( size == 0 )
	{
		assert( false );
		size = 1;
	}
	return WTF::fastMalloc( size, GLOBALNEW_MEMORY_TAG );
}


void *operator new[]( size_t size )
{
	// http://www.cplusplus.com/reference/new/operator%20new/
	// If this argument is zero, the function still returns a distinct non-null pointer on success
	// (although dereferencing this pointer leads to undefined behavior).
	if ( size == 0 )
	{
		assert( false );
		size = 1;
	}
	return WTF::fastMalloc( size, GLOBALNEW_MEMORY_TAG );
}


void operator delete( void *p )
{
	if ( p == nullptr )
	{
		return;
	}
	WTF::fastFree( p );
}


void operator delete[]( void *p )
{
	if ( p == nullptr )
	{
		return;
	}
	WTF::fastFree( p );
}


// STL does calls operator( size, nothrow ); e.g: xmemory Line 30.
void *operator new( size_t size, const std::nothrow_t & ) noexcept
{
	// http://www.cplusplus.com/reference/new/operator%20new/
	// If this argument is zero, the function still returns a distinct non-null pointer on success
	// (although dereferencing this pointer leads to undefined behavior).
	if ( size == 0 )
	{
		assert( false );
		size = 1;
	}
	return WTF::fastMalloc( size, GLOBALNEW_MEMORY_TAG );
}


void *operator new[]( size_t size, const std::nothrow_t & ) noexcept
{
	// http://www.cplusplus.com/reference/new/operator%20new/
	// If this argument is zero, the function still returns a distinct non-null pointer on success
	// (although dereferencing this pointer leads to undefined behavior).
	if ( size == 0 )
	{
		assert( false );
		size = 1;
	}
	return WTF::fastMalloc( size, GLOBALNEW_MEMORY_TAG );
}


void operator delete( void *p, const std::nothrow_t & )
{
	if ( p == nullptr )
	{
		return;
	}
	WTF::fastFree( p );
}


void operator delete[]( void *p, const std::nothrow_t & )
{
	if ( p == nullptr )
	{
		return;
	}
	WTF::fastFree( p );
}


void *operator new( std::size_t size, std::align_val_t align )
{
	// http://www.cplusplus.com/reference/new/operator%20new/
	// If this argument is zero, the function still returns a distinct non-null pointer on success
	// (although dereferencing this pointer leads to undefined behavior).
	if ( size == 0 )
	{
		assert( false );
		size = 1;
	}
	return WTF::fastAlignedMalloc( static_cast<size_t>( align ), size, GLOBALNEW_MEMORY_TAG );
}


void *operator new[]( std::size_t size, std::align_val_t align )
{
	// http://www.cplusplus.com/reference/new/operator%20new/
	// If this argument is zero, the function still returns a distinct non-null pointer on success
	// (although dereferencing this pointer leads to undefined behavior).
	if ( size == 0 )
	{
		assert( false );
		size = 1;
	}
	return WTF::fastAlignedMalloc( static_cast<size_t>( align ), size, GLOBALNEW_MEMORY_TAG );
}


void *operator new( std::size_t size, std::align_val_t align, const std::nothrow_t & ) noexcept
{
	// http://www.cplusplus.com/reference/new/operator%20new/
	// If this argument is zero, the function still returns a distinct non-null pointer on success
	// (although dereferencing this pointer leads to undefined behavior).
	if ( size == 0 )
	{
		assert( false );
		size = 1;
	}
	return WTF::fastAlignedMalloc( static_cast<size_t>( align ), size, GLOBALNEW_MEMORY_TAG );
}


void *operator new[]( std::size_t size, std::align_val_t align, const std::nothrow_t & ) noexcept
{
	// http://www.cplusplus.com/reference/new/operator%20new/
	// If this argument is zero, the function still returns a distinct non-null pointer on success
	// (although dereferencing this pointer leads to undefined behavior).
	if ( size == 0 )
	{
		assert( false );
		size = 1;
	}
	return WTF::fastAlignedMalloc( static_cast<size_t>( align ), size, GLOBALNEW_MEMORY_TAG );
}


void operator delete( void *p, std::align_val_t ) noexcept
{
	if ( p == nullptr )
	{
		return;
	}
	WTF::fastAlignedFree( p );
}


void operator delete[]( void *p, std::align_val_t ) noexcept
{
	if ( p == nullptr )
	{
		return;
	}
	WTF::fastAlignedFree( p );
}


void operator delete( void *p, std::align_val_t, const std::nothrow_t & ) noexcept
{
	if ( p == nullptr )
	{
		return;
	}
	WTF::fastAlignedFree( p );
}


void operator delete[]( void *p, std::align_val_t, const std::nothrow_t & ) noexcept
{
	if ( p == nullptr )
	{
		return;
	}
	WTF::fastAlignedFree( p );
}
