/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "PlatformFeatureDefs.h"
#include "Universal/TelescopeLog.h"
#include "TelescopeLib.h"
#include "WebView.h"
#include "ClientWrappers/MemoryAllocatorBase.h"
#include "ClientWrappers/CURLReplacement/CurlReplacementImpl.h"
#include "ClientWrappers/HashingReplacement/HashingReplacement.h"
#include "ClientWrappers/ICUMMAPReplacement/IcuMmpReplacement.h"
#include "ClientWrappers/SystemAPIReplacement/SystemAPIReplacement.h"
#include "ClientWrappers/ErrorHandlingReplacement/ErrorHandlingReplacement.h"
#include "ClientWrappers/LogReplacement/LogReplacement.h"
#include "ClientWrappers/MediaPlayerReplacement/MediaPlayerReplacement.h"
#include "wtf/MainThread.h"
#include "wtf/RunLoop.h"
#include "wtf/TelescopePrintStream.h"
#include "fontconfig/fontconfig.h"
#include <WebCore/MainThreadSharedTimer.h>
#include "Logging.h"
#include <wtf/ThreadSpecific.h>
#include <wtf/WorkQueue.h>
#include "network/curl/CurlShutdown.h"
#include <WebCore/NetworkStateNotifier.h>
#include <WebCore/CurlCacheManager.h>
#include <WebCore/MemoryRelease.h>
#include "MemoryTrackingScope.h"
#include "Universal/TelescopeLog.h"
#include <JavaScriptCore/RemoteInspectorServer.h>
#include <bitset>
#include <WebCore/MemoryCache.h>
extern void InstallCompressionWrapper( CompressionWrapperBase *pCompressionWrapper );
extern CompressionWrapperBase *getCompressionWrapper();

#if TELESCOPE_PREDEFINE_DEBUG
#define sPublishVersion "Debug"
#elif TELESCOPE_PREDEFINE_RELEASE // #if TELESCOPE_PREDEFINE_DEBUG
#define sPublishVersion "Release"
#elif TELESCOPE_PREDEFINE_SHIP // #elif TELESCOPE_PREDEFINE_RELEASE // #if TELESCOPE_PREDEFINE_DEBUG
#define sPublishVersion "Ship"
#else // #elif TELESCOPE_PREDEFINE_SHIP // #elif TELESCOPE_PREDEFINE_RELEASE // #if TELESCOPE_PREDEFINE_DEBUG
#define sPublishVersion "Unknown"
#endif // #else // #elif TELESCOPE_PREDEFINE_SHIP // #elif TELESCOPE_PREDEFINE_RELEASE // #if TELESCOPE_PREDEFINE_DEBUG

namespace Telescope
{
static std::bitset<32> m_featureFlag{};

bool getConfigFlagWrapper( TelescopeFeature enumIndex )
{
	return m_featureFlag.test( (size_t)enumIndex );
}


void setConfigFlagWrapper( TelescopeFeature enumIndex, bool value )
{
	m_featureFlag.set( (size_t)enumIndex, value );
}


void TelescopeLib::PreInit( TelescopeConfig *config )
{
	// Memory must be the first one to install
	InstallMemoryAllocator( config->m_pMemAllocator );

	InstallErrorHandlingWrapper( config->m_pErrorHandlingImpl );

	ASSERT_WITH_MESSAGE( config->m_pHttpWrapper, "Telescope must have the implementation of HTTP." );
	if ( config->m_pHttpWrapper )
	{
		InstallHttpWrapper( config->m_pHttpWrapper );
	}

	ASSERT_WITH_MESSAGE( config->m_pHashingImpl, "Telescope must have the implementation of HASHING." );
	if ( config->m_pHashingImpl )
	{
		InstallHashingWrapper( config->m_pHashingImpl );
	}

	ASSERT_WITH_MESSAGE( config->m_pCompressionImpl, "Telescope must have the implementation of a Compression implementation." );
	if ( config->m_pCompressionImpl )
	{
		InstallCompressionWrapper( config->m_pCompressionImpl );
	}

	ASSERT_WITH_MESSAGE( config->m_pAssetLoaderImpl, "Telescope must have the implementation to asset loader." );
	if ( config->m_pAssetLoaderImpl )
	{
		InstallOpenFileWrapper( config->m_pAssetLoaderImpl );
	}

	ASSERT_WITH_MESSAGE( config->m_pSysAPIImpl, "Telescope must have the implementation of system API." );
	if ( config->m_pSysAPIImpl )
	{
		InstallSystemAPIWrapper( config->m_pSysAPIImpl );
	}

	ASSERT_WITH_MESSAGE( config->m_pLogImpl, "Telescope must have the implementation of log system." );
	if ( config->m_pLogImpl )
	{
		InstallLogWrapper( config->m_pLogImpl );
	}

	LogInfo( Telescope::LogChannel::GENERAL, "Telescope video %s", config->m_pMediaPlayerWrapperClientFactory ? "enabled" : "disabled" );
	if ( config->m_pMediaPlayerWrapperClientFactory )
	{
		InstallMediaPlayerWrapperFactory( config->m_pMediaPlayerWrapperClientFactory );
	}
}

const char *GetChangelistDescription();

unsigned GetChangelistNumber()
{
	static unsigned cl = 0;

	if ( cl == 0 )
	{
		char dummy[32];
		sscanf( GetChangelistDescription(), "%s %s %u", dummy, dummy, &cl );
	}

	return cl;
}


bool TelescopeLib::Init( TelescopeConfig *config )
{
	WTF::Thread::initMainThreadHolder();
	WTF::RunLoop::initializeMainRunLoop();

	RELEASE_LOG_INFO( Telescope, "PublishVersion:[%s], Compile Time:[%s %s], CL: %u\n",
		sPublishVersion, __DATE__, __TIME__, GetChangelistNumber() );

	return true;
}


void TelescopeLib::Shutdown( bool force )
{
	// TODO: Looks like Telescope is not shutdown when reaches here.
	// Telescope_UninstallMemoryAllocator();

#if ENABLE( REMOTE_INSPECTOR )
	if ( getConfigFlagWrapper( TelescopeFeature::RemoteInspector ) )
	{
		Inspector::RemoteInspectorServer::singleton().shutdown();
		Inspector::RemoteInspectorSocketEndpoint::singleton().shutdown();
	}
#endif // #if ENABLE( REMOTE_INSPECTOR )

	WebCore::CurlCacheManager::singleton().saveIndex();

#if TELESCOPE_USING( TELESCOPE_MICROSOFT_PROGRAM )
	NetworkStateNotifier::singleton().Cleanup();
#endif // #if TELESCOPE_USING( TELESCOPE_MICROSOFT_PROGRAM )

	WTF::RunLoop::shutdown();

#if USE( PTHREADS )
	WTF::shutdownPThreadKeyHashMap();
#endif // #if USE( PTHREADS )

	WebCore::curlShutdown();

	if ( force )
	{
		WTF::Thread::killAllThreads();
	}

	WTF::WorkQueue::stopAllWorkQueues();

	WTF::Thread::waitAllThreads();
}


void TelescopeLib::Tick()
{
	WTF::dispatchFunctionsFromMainThread();

	WTF::Thread::cleanStaledThread();

	WTF::RunLoop::main().iterate();
}


WebView *TelescopeLib::CreateView()
{
	TELESCOPE_MEMORY_SCOPE( "TelescopeLib::CreateView" );

	return new WebView();
}


void TelescopeLib::DestroyView( WebView *view )
{
	view->ShutdownView();
	delete view;
}


bool TelescopeLib::InstallFont( const unsigned char *fontData, size_t dataSize )
{
	return FcAddFontMemory( fontData, dataSize );
}


void TelescopeLib::InitFileCache( const char *cacheDirectory, size_t storageSizeLimitInBytes )
{
	String cacheFolder( cacheDirectory );
	WebCore::CurlCacheManager::singleton().setCacheDirectory( cacheFolder );
	WebCore::CurlCacheManager::singleton().setStorageSizeLimit( storageSizeLimitInBytes );
}


void TelescopeLib::InitRemoteDebugPort( uint16_t port )
{
#if ENABLE( REMOTE_INSPECTOR )
	setConfigFlagWrapper( TelescopeFeature::RemoteInspector, true );

	if ( Inspector::RemoteInspectorServer::singleton().start( "0.0.0.0", port ) )
	{
		Telescope::LogInfo( Telescope::LogChannel::GENERAL, "Remote Inspector listen port is %u", port );
	}
	else
	{
		Telescope::LogError( Telescope::LogChannel::GENERAL, "Remote Inspector listen port %u failed", port );
	}
#endif // #if ENABLE( REMOTE_INSPECTOR )
}

unsigned TelescopeLib::GetChangelistNumber() const
{
	return Telescope::GetChangelistNumber();
}


void TelescopeLib::GCNow()
{
	WebCore::releaseMemoryCacheMemory();
}

void TelescopeLib::DumpMemoryCacheUsage()
{
	WebCore::MemoryCache::singleton().dump();
}
} // namespace Telescope
