/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once
#include "PlatformFeatureDefs.h"
#include "WebViewGroup.h"
#include "TelescopeJavascriptValue.h"
#include "TelescopeAPI.h"
#include "WebCore/IntRect.h"
#include "WebFrame.h"
#include "wtf/StdUnorderedMap.h"

#if USE( TEXTURE_MAPPER_TELESCOPE )
#include "AcceleratedCompositingContext.h"
#endif // #if USE( TEXTURE_MAPPER_TELESCOPE )
using namespace WebCore;

namespace Telescope
{
constexpr int max_dirty_regions = 50;

class DirtyRegions
{
	WTF_MAKE_FAST_ALLOCATED;

public:
	DirtyRegions()
		: mNumRegions( 0 )
		, mRegionsOverflow( false )
	{};

	void AddRegion( const WebCore::IntRect *region )
	{
		if ( mNumRegions == max_dirty_regions )
		{
			mRegionsOverflow = true;
		}
		else
		{
			// If the region was contained by other region ignore it
			// if the region is intersected with other region merge the region
			// and go through all the existing regions and merge intersected region.
			for ( unsigned i = 0; i < mNumRegions; i++ )
			{
				WebCore::IntRect &rect = mRegions[i];
				if ( rect.contains( *region ) )
				{
					return;
				}
				if ( ShouldRegionsMerged( &rect, region ) )
				{
					rect.unite( *region );

					MergeIntersectedRegions();
					return;
				}
			}

			mRegions[mNumRegions++] = *region;
		}
	}

	void Clear()
	{
		mNumRegions = 0;
		mRegionsOverflow = false;
	}

	unsigned Count()
	{
		return mNumRegions;
	}

	bool IsOverflow()
	{
		return mRegionsOverflow;
	}

	const WebCore::IntRect *GetRegion( unsigned index )
	{
		// if dirty region overflow, trigger a fullscreen redraw
		ASSERT( index < mNumRegions && !mRegionsOverflow );
		return &mRegions[index];
	}

	bool Empty()
	{
		return mNumRegions == 0;
	}

private:
	void Erase( const int index )
	{
		ASSERT( index < mNumRegions );
		if ( index != mNumRegions - 1 )
		{
			memcpy( mRegions + index, mRegions + index + 1, sizeof( WebCore::IntRect ) * ( mNumRegions - index - 1 ) );
		}
		mNumRegions--;
	}

	bool ShouldRegionsMerged( const WebCore::IntRect *left, const WebCore::IntRect *right )
	{
		// merge intersected regions
		if ( left->intersects( *right ) )
		{
			return true;
		}
		else
		{
			// if both region shares a same border merge the region
			//         ------------
			//         |          |
			//-------------------------------
			//|        |    A     |         |
			//-------------------------------
			//         |          |
			//         ------------
			if ( left->x() == right->x() && left->maxX() == right->maxX()
				&& ( left->y() == right->maxY() || left->maxY() == right->y() ) )
			{
				return true;
			}

			if ( left->y() == right->y() && left->maxY() == right->maxY()
				&& ( left->x() == right->maxX() || left->maxX() == right->x() ) )
			{
				return true;
			}
		}

		return false;
	}

	void MergeIntersectedRegions()
	{
		const unsigned count = Count();
		for ( int i = 0; i < count; i++ )
		{
			WebCore::IntRect &rect = mRegions[i];
			for ( int j = i + 1; j < count; j++ )
			{
				WebCore::IntRect &rRect = mRegions[j];
				if ( ShouldRegionsMerged( &rect, &rRect ) )
				{
					rect.unite( rRect );
					Erase( j );
					MergeIntersectedRegions();
					return;
				}
			}
		}
	}

	WebCore::IntRect mRegions[max_dirty_regions];
	unsigned mNumRegions;
	bool mRegionsOverflow;
};


struct ViewPrivate
{
	WTF_MAKE_FAST_ALLOCATED;

public:
	friend class ViewIME;

	ViewPrivate()
		: mPage( nullptr )
		, mInitialized( false )
		, mWebViewGroup( WebViewGroup::getOrCreate( String(), String( "" ) ) )
		, mWidth( 0 )
		, mHeight( 0 )
		, mTransparent( true )
		, mMainFrame( nullptr )
#if USE( TEXTURE_MAPPER_TELESCOPE )
		  ,
		  mIsAcceleratedCompositing( false )
		, mCompostingContext( nullptr )
		, mRender( nullptr )
#endif // #if USE( TEXTURE_MAPPER_TELESCOPE )
	{
	}
	WebCore::Page *mPage;
	bool mInitialized;
	Ref<WebViewGroup> mWebViewGroup;
	int mWidth;
	int mHeight;
	bool mTransparent;
	WebFrame *mMainFrame;
	Surface *mDisplaySurface;
	IntRect mWindowRect;
	WTF::StdUnorderedMap<const char *, Telescope::IJSBoundObject *> mBoundObjects;
	DirtyRegions mDirtyRegions;

#if USE( TEXTURE_MAPPER_TELESCOPE )
	bool mIsAcceleratedCompositing;
	AcceleratedCompositingContext *mCompostingContext;
	HardwareRenderBase *mRender;
#endif // #if USE( TEXTURE_MAPPER_TELESCOPE )
};
} // namespace Telescope
