/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "TelescopeMediaPlayerPrivate.h"
#if ENABLE( MEDIA_SOURCE )
#include "TelescopeMediaSourcePrivate.h"
#endif // #if ENABLE( MEDIA_SOURCE )

#include <WebCore/NotImplemented.h>
#include <WebCore/FrameView.h>
#include <WebCore/CachedResourceLoader.h>
#if ENABLE( MEDIA_SOURCE )
#include <WebCore/MediaSourcePrivateClient.h>
#include <WebCore/MediaSourcePrivate.h>
#endif // #if ENABLE( MEDIA_SOURCE )

#if USE( CAIRO )
#include <WebCore/CairoOperations.h>
#include <WebCore/PlatformContextCairo.h>
#include <cairo.h>
#endif // #if USE( CAIRO )

namespace Telescope
{
class TelescopeMediaPlayerFactory final : public WebCore::MediaPlayerFactory
{
public:
	static void registerMediaEngine( WebCore::MediaEngineRegistrar registrar )
	{
		registrar( makeUnique<TelescopeMediaPlayerFactory>() );
	}

	static WebCore::MediaPlayer::SupportsType supportsType( const WebCore::MediaEngineSupportParameters &parameters )
	{
		if ( parameters.type.isEmpty() )
		{
			return WebCore::MediaPlayer::SupportsType::IsNotSupported;
		}

		if ( mimeTypeCache().contains( parameters.type.containerType() ) )
		{
			return WebCore::MediaPlayer::SupportsType::IsSupported;
		}

		return WebCore::MediaPlayer::SupportsType::IsNotSupported;
	}

private:
	static const HashSet<String, ASCIICaseInsensitiveHash> &mimeTypeCache()
	{
		static NeverDestroyed<HashSet<String, ASCIICaseInsensitiveHash>> cachedTypes(
			std::initializer_list<String>
			{
				"video/mp4",
				"audio/mp4",
				"video/x-m4v",
				"audio/x-m4v",
				"video/quicktime"
			}
			);

		return cachedTypes;
	}

	WebCore::MediaPlayerEnums::MediaEngineIdentifier identifier() const final
	{
		return WebCore::MediaPlayerEnums::MediaEngineIdentifier::MediaFoundation;
	}

	std::unique_ptr<WebCore::MediaPlayerPrivateInterface> createMediaEnginePlayer( WebCore::MediaPlayer *pPlayer ) const final
	{
		if ( !pPlayer )
		{
			return nullptr;
		}

		return makeUnique<TelescopeMediaPlayerPrivate>( *pPlayer );
	}

	void getSupportedTypes( HashSet<String, ASCIICaseInsensitiveHash> &types ) const final
	{
		types = mimeTypeCache();
	}

	WebCore::MediaPlayer::SupportsType supportsTypeAndCodecs( const WebCore::MediaEngineSupportParameters &parameters ) const final
	{
		return supportsType( parameters );
	}
};

void InstallTelescopeMediaEngine()
{
	WebCore::MediaPlayerFactorySupport::callRegisterMediaEngine( TelescopeMediaPlayerFactory::registerMediaEngine );
}


bool IsSupportsTypeAndCodecs( const WebCore::MediaEngineSupportParameters &parameters )
{
	return TelescopeMediaPlayerFactory::supportsType( parameters ) != WebCore::MediaPlayer::SupportsType::IsNotSupported;
}


void TelescopeMediaPlayerPrivate::load( const String &url )
{
	m_pClientMediaPlayer->Load( url.ascii().data() );
	m_pClientMediaPlayer->Loop( m_player->isLooping() );
	m_pClientMediaPlayer->Mute( m_player->muted() );

	m_networkState = WebCore::MediaPlayer::NetworkState::Loading;
	m_player->networkStateChanged();

	m_readyState = WebCore::MediaPlayer::ReadyState::HaveNothing;
	m_player->readyStateChanged();

	m_mediaUpdateTimer.startRepeating( 0_ms );
}

#if ENABLE( MEDIA_SOURCE )
void TelescopeMediaPlayerPrivate::load( const String &, WebCore::MediaSourcePrivateClient *pClient )
{
	pClient->setPrivateAndOpen( TelescopeMediaSourcePrivate::create( this, pClient ) );
}


void TelescopeMediaPlayerPrivate::appendData( Vector<unsigned char> &&data, const WebCore::ContentType &contentType )
{
	// TODO:
}
#endif // #if ENABLE( MEDIA_SOURCE )

void TelescopeMediaPlayerPrivate::cancelLoad()
{
	notImplemented();
}


void TelescopeMediaPlayerPrivate::play()
{
	m_pClientMediaPlayer->Play( true );
	m_bPaused = false;
}


void TelescopeMediaPlayerPrivate::pause()
{
	m_pClientMediaPlayer->Play( false );
	m_bPaused = true;
}


bool TelescopeMediaPlayerPrivate::paused() const
{
	return m_bPaused;
}


bool TelescopeMediaPlayerPrivate::supportsFullscreen() const
{
	notImplemented();
	return true;
}


WebCore::FloatSize TelescopeMediaPlayerPrivate::naturalSize() const
{
	if ( m_pClientMediaPlayer->IsAvailable() && m_pClientMediaPlayer->HasVideo() )
	{
		WebCore::FloatSize size;
		size_t width = 0;
		size_t height = 0;
		m_pClientMediaPlayer->GetVideoSize( width, height );
		size.setWidth( static_cast<float>( width ) );
		size.setHeight( static_cast<float>( height ) );
		return size;
	}
	return WebCore::FloatSize();
}


void TelescopeMediaPlayerPrivate::setSize( const WebCore::IntSize &size )
{
	UNUSED_VAR( size );
}


bool TelescopeMediaPlayerPrivate::hasVideo() const
{
	return m_pClientMediaPlayer->IsAvailable() && m_pClientMediaPlayer->HasVideo();
}


bool TelescopeMediaPlayerPrivate::hasAudio() const
{
	return m_pClientMediaPlayer->IsAvailable() && m_pClientMediaPlayer->HasAudio();
}


void TelescopeMediaPlayerPrivate::setVisible( bool visible )
{
	notImplemented();
}


bool TelescopeMediaPlayerPrivate::seeking() const
{
	// We assume seeking is immediately complete.
	return false;
}


void TelescopeMediaPlayerPrivate::seek( float time )
{
	m_pClientMediaPlayer->Seek( time );
	m_player->timeChanged();
}


void TelescopeMediaPlayerPrivate::setRate( float )
{
	notImplemented();
}


float TelescopeMediaPlayerPrivate::duration() const
{
	return m_pClientMediaPlayer->Duration();
}


float TelescopeMediaPlayerPrivate::currentTime() const
{
	float t = m_pClientMediaPlayer->CurrentTime();
	return t >= 0 ? t : 0.0f;
}


void TelescopeMediaPlayerPrivate::setVolume( float )
{
	notImplemented();
}


void TelescopeMediaPlayerPrivate::setMuted( bool bMute )
{
	m_pClientMediaPlayer->Mute( bMute );
}


WebCore::MediaPlayer::NetworkState TelescopeMediaPlayerPrivate::networkState() const
{
	return m_networkState;
}


WebCore::MediaPlayer::ReadyState TelescopeMediaPlayerPrivate::readyState() const
{
	return m_readyState;
}


float TelescopeMediaPlayerPrivate::maxTimeSeekable() const
{
	return duration();
}


std::unique_ptr<WebCore::PlatformTimeRanges> TelescopeMediaPlayerPrivate::buffered() const
{
	auto ranges = makeUnique<WebCore::PlatformTimeRanges>();
	ranges->add( MediaTime::zeroTime(), MediaTime::createWithDouble( currentTime() ) );
	return ranges;
}


bool TelescopeMediaPlayerPrivate::didLoadingProgress() const
{
	return m_pClientMediaPlayer->IsAvailable();
}


void TelescopeMediaPlayerPrivate::paint( WebCore::GraphicsContext &context, const WebCore::FloatRect &rect )
{
	const MediaPlayerWrapperBase::RGBBuffer &rgbBuffer = m_pClientMediaPlayer->GetVideoRGBBuffer();

	if ( !rgbBuffer.IsValid() )
	{
		return;
	}

	auto pRGBBuffer = static_cast<unsigned char *>( const_cast<void *>( rgbBuffer.pRGBBuffer ) );
	cairo_surface_t *pImage = cairo_image_surface_create_for_data( pRGBBuffer, CAIRO_FORMAT_RGB24, rgbBuffer.bufferWidth, rgbBuffer.bufferHeight, rgbBuffer.bufferRowPitch );
	if ( pImage )
	{
		WebCore::FloatRect srcRect( 0, 0, rgbBuffer.bufferWidth, rgbBuffer.bufferHeight );
		WebCore::FloatRect destRect( rect.x(), rect.y(), rect.width(), rect.height() );
		ASSERT( context.hasPlatformContext() );
		auto &state = context.state();
		auto interpolationQuality = state.imageInterpolationQuality;
		if ( rgbBuffer.bufferWidth != rect.width() || rgbBuffer.bufferHeight != rect.height() )
		{
			interpolationQuality = WebCore::InterpolationQuality::Low;
		}
		WebCore::Cairo::drawSurface( *context.platformContext(), pImage, destRect, srcRect, interpolationQuality, state.alpha, WebCore::Cairo::ShadowState( state ) );
		cairo_surface_destroy( pImage );
	}
}


TelescopeMediaPlayerPrivate::TelescopeMediaPlayerPrivate( WebCore::MediaPlayer &pPlayer )
	: m_player( &pPlayer )
	, m_bPaused( false )
	, m_pClientMediaPlayer( nullptr )
	, m_networkState( WebCore::MediaPlayer::NetworkState::Empty )
	, m_readyState( WebCore::MediaPlayer::ReadyState::HaveNothing )
	, m_mediaUpdateTimer( RunLoop::current(), this, &TelescopeMediaPlayerPrivate::Update )
{
	MediaPlayerWrapperClientFactoryBase *pMediaPlayerWrapperClientFactory = GetMediaPlayerWrapperFactory();
	assert( pMediaPlayerWrapperClientFactory );

	m_pClientMediaPlayer = pMediaPlayerWrapperClientFactory->CreateMediaPlayerWrapper( *this );
	assert( m_pClientMediaPlayer );
}


TelescopeMediaPlayerPrivate::~TelescopeMediaPlayerPrivate()
{
	m_mediaUpdateTimer.stop();

	assert( m_pClientMediaPlayer );

	MediaPlayerWrapperClientFactoryBase *pMediaPlayerWrapperClientFactory = GetMediaPlayerWrapperFactory();
	assert( pMediaPlayerWrapperClientFactory );

	pMediaPlayerWrapperClientFactory->DestroyMediaPlayerWrapper( m_pClientMediaPlayer );

	auto pClient = client();
	if ( pClient )
	{
		pClient->platformLayerWillBeDestroyed();
	}
}


void TelescopeMediaPlayerPrivate::OnMetadataReady()
{
	m_readyState = WebCore::MediaPlayer::ReadyState::HaveMetadata;
	m_player->readyStateChanged();
}


void TelescopeMediaPlayerPrivate::OnReadyPlay()
{
	m_networkState = WebCore::MediaPlayer::NetworkState::Loading;
	m_player->networkStateChanged();

	m_readyState = WebCore::MediaPlayer::ReadyState::HaveEnoughData;
	m_player->readyStateChanged();
	m_frameRendered = true;
}


void TelescopeMediaPlayerPrivate::OnEndPlay()
{
	m_bPaused = true;
	m_player->playbackStateChanged();
}


void TelescopeMediaPlayerPrivate::OnFatalError()
{
	m_networkState = WebCore::MediaPlayer::NetworkState::NetworkError;
	m_player->networkStateChanged();
}


void TelescopeMediaPlayerPrivate::OnNewVideoFrame()
{
	m_player->repaint();

	auto pClient = client();
	if ( pClient )
	{
		pClient->setPlatformLayerNeedsDisplay();
	}
}


void TelescopeMediaPlayerPrivate::Update()
{
	m_pClientMediaPlayer->Update();
}

#if ENABLE( DEVGUI )
void TelescopeMediaPlayerPrivate::DrawDebugInfo( cairo_t *cairoContext, DevGUIContext &guiContext )
{
	if ( m_pClientMediaPlayer )
	{
		int totalFrames, droppedFrames;
		m_pClientMediaPlayer->GetDebugInfo( totalFrames, droppedFrames );
		guiContext.currentY = guiContext.currentY + guiContext.fontSize;
		cairo_move_to( cairoContext, 0, guiContext.currentY );
		char debugInfo[255] = { 0 };
		sprintf_s( debugInfo, "Video frame drop/total: %d / %d", droppedFrames, totalFrames );
		cairo_show_text( cairoContext, debugInfo );
	}
}
#endif // #if ENABLE( DEVGUI )

void TelescopeMediaPlayerPrivate::paintToTextureMapper( WebCore::TextureMapper &textureMapper, const WebCore::FloatRect &rect, const WebCore::TransformationMatrix &modelViewMatrix, float opacity )
{
	UNUSED_VAR( opacity );
	if ( m_frameRendered && supportsAcceleratedRendering() )
	{
		size_t videoWidth, videoHeight;
		m_pClientMediaPlayer->GetVideoSize(videoWidth, videoHeight);
		WebCore::FloatRect targetRect(0, 0, static_cast<float>( videoWidth ), static_cast<float>( videoHeight ) );
		textureMapper.drawVideo( modelViewMatrix.mapRect( rect ), targetRect );
	}
}

bool TelescopeMediaPlayerPrivate::supportsAcceleratedRendering() const
{
	return m_pClientMediaPlayer->AcceleratedRenderingEnabled();
}
} // namespace Telescope
