/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include <WebCore/MediaPlayerPrivate.h>
#include <webcore/TextureMapperPlatformLayer.h>
#include <ClientWrappers/MediaPlayerReplacement/MediaPlayerReplacement.h>
#include <wtf/RunLoop.h>
#include "devgui/DevGUI.h"
namespace Telescope
{
class MediaPlayerWrapperBase;

bool IsSupportsTypeAndCodecs( const WebCore::MediaEngineSupportParameters &parameters );

class TelescopeMediaPlayerPrivate final : public WebCore::MediaPlayerPrivateInterface, public CanMakeWeakPtr<TelescopeMediaPlayerPrivate>, public MediaPlayerWrapperEventHandler
#if ENABLE( DEVGUI )
	,
										  public DevGUIClient
#endif // #if ENABLE( DEVGUI )
	,public WebCore::TextureMapperPlatformLayer
{
	WTF_MAKE_FAST_ALLOCATED;

public:
	TelescopeMediaPlayerPrivate( WebCore::MediaPlayer &pPlayer );
	~TelescopeMediaPlayerPrivate();

	void load( const String &url ) final;
#if ENABLE( MEDIA_SOURCE )
	void load( const String &url, WebCore::MediaSourcePrivateClient * ) final;
	void appendData( Vector<unsigned char> &&, const WebCore::ContentType & );
#endif // #if ENABLE( MEDIA_SOURCE )
	void cancelLoad() final;
	void play() final;
	void pause() final;
	bool paused() const final;
	bool supportsFullscreen() const final;
	WebCore::FloatSize naturalSize() const final;
	void setSize( const WebCore::IntSize & ) final;
	bool hasVideo() const final;
	bool hasAudio() const final;
	void setVisible( bool ) final;
	bool seeking() const final;
	void seek( float ) final;
	void setRate( float ) final;
	float duration() const final;
	float currentTime() const final;
	void setVolume( float ) final;
	void setMuted( bool ) final;
	WebCore::MediaPlayer::NetworkState networkState() const final;
	WebCore::MediaPlayer::ReadyState readyState() const final;
	float maxTimeSeekable() const final;
	std::unique_ptr<WebCore::PlatformTimeRanges> buffered() const final;
	bool didLoadingProgress() const final;
	void paint( WebCore::GraphicsContext &, const WebCore::FloatRect & ) final;
	void Update();

private:
	// MediaPlayerWrapperEventHandler implementation
	void OnMetadataReady() override;
	void OnReadyPlay() override;
	void OnEndPlay() override;
	void OnFatalError() override;
	void OnNewVideoFrame() override;

	bool requiresImmediateCompositing() const final { return true; }
	bool supportsAcceleratedRendering() const final;

	WebCore::PlatformLayer* platformLayer() const override { return const_cast<TelescopeMediaPlayerPrivate*>(this); }
	void paintToTextureMapper( WebCore::TextureMapper&, const WebCore::FloatRect&, const WebCore::TransformationMatrix& modelViewMatrix = WebCore::TransformationMatrix(), float opacity = 1.0 ) final;

#if ENABLE( DEVGUI )
	void DrawDebugInfo( cairo_t *cairoContext, DevGUIContext &guiContext ) override;
#endif // #if ENABLE( DEVGUI )
private:
	WebCore::MediaPlayer *m_player;
	bool m_bPaused;
	MediaPlayerWrapperBase *m_pClientMediaPlayer;

	WebCore::MediaPlayer::NetworkState m_networkState;
	WebCore::MediaPlayer::ReadyState m_readyState;

	RunLoop::Timer<TelescopeMediaPlayerPrivate> m_mediaUpdateTimer;
	bool m_frameRendered{ false };
};

} // namespace Telescope
