/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "ViewIME.h"

#if TELESCOPE_USING( TELESCOPE_PC_PROGRAM )
#include "WebCore/Editor.h"
#include "WebView.h"
#include "Universal/ViewPrivate.h"
#include "WebFrame.h"
#include "WebCore/Page.h"
#include "WebCore/FocusController.h"
#include "WebCore/Frame.h"
#include "WebCore/FrameView.h"

using namespace WebCore;

namespace Telescope
{
static bool getCompositionString( HIMC hInputContext, DWORD type, String &result )
{
	int compositionLength = IMMDict::dict().getCompositionString( hInputContext, type, 0, 0 );
	if ( compositionLength <= 0 )
	{
		return false;
	}
	Vector<UChar> compositionBuffer( compositionLength / 2 );
	compositionLength = IMMDict::dict().getCompositionString( hInputContext, type, (LPVOID)compositionBuffer.data(), compositionLength );
	result = String( compositionBuffer.data(), compositionLength / 2 );
	ASSERT( !compositionLength || compositionBuffer[0] );
	ASSERT( !compositionLength || compositionBuffer[compositionLength / 2 - 1] );
	return true;
}


static void compositionToUnderlines( const Vector<DWORD> &clauses, const Vector<BYTE> &attributes, Vector<CompositionUnderline> &underlines )
{
	if ( clauses.isEmpty() )
	{
		underlines.clear();
		return;
	}

	const size_t numBoundaries = clauses.size() - 1;
	underlines.resize( numBoundaries );
	for ( unsigned i = 0; i < numBoundaries; i++ )
	{
		underlines[i].startOffset = clauses[i];
		underlines[i].endOffset = clauses[i + 1];
		BYTE attribute = attributes[clauses[i]];
		underlines[i].thick = attribute == ATTR_TARGET_CONVERTED || attribute == ATTR_TARGET_NOTCONVERTED;
		underlines[i].color = Color( 0, 0, 0 );
	}
}


const IMMDict &IMMDict::dict()
{
	static IMMDict instance;
	return instance;
}


IMMDict::IMMDict()
{
	m_instance = ::LoadLibraryW( L"IMM32.DLL" );
	getContext = reinterpret_cast<getContextPtr>( ::GetProcAddress( m_instance, "ImmGetContext" ) );
	ASSERT( getContext );
	releaseContext = reinterpret_cast<releaseContextPtr>( ::GetProcAddress( m_instance, "ImmReleaseContext" ) );
	ASSERT( releaseContext );
	getCompositionString = reinterpret_cast<getCompositionStringPtr>( ::GetProcAddress( m_instance, "ImmGetCompositionStringW" ) );
	ASSERT( getCompositionString );
	setCandidateWindow = reinterpret_cast<setCandidateWindowPtr>( ::GetProcAddress( m_instance, "ImmSetCandidateWindow" ) );
	ASSERT( setCandidateWindow );
	setOpenStatus = reinterpret_cast<setOpenStatusPtr>( ::GetProcAddress( m_instance, "ImmSetOpenStatus" ) );
	ASSERT( setOpenStatus );
	notifyIME = reinterpret_cast<notifyIMEPtr>( ::GetProcAddress( m_instance, "ImmNotifyIME" ) );
	ASSERT( notifyIME );
	associateContextEx = reinterpret_cast<associateContextExPtr>( ::GetProcAddress( m_instance, "ImmAssociateContextEx" ) );
	ASSERT( associateContextEx );
}


ViewIME::ViewIME()
{
}


ViewIME::~ViewIME()
{
}


HIMC ViewIME::getIMMContext()
{
	HIMC context = IMMDict::dict().getContext( m_viewWindow );
	return context;

	return nullptr;
}


void ViewIME::releaseIMMContext( HIMC )
{
}


void ViewIME::prepareCandidateWindow( Frame *targetFrame, HIMC hInputContext )
{
}


float ViewIME::deviceScaleFactor() const
{
	return 1.0f;
}


void ViewIME::SetViewWnd( HWND wnd )
{
	m_viewWindow = wnd;
}


LRESULT ViewIME::onIMERequestCharPosition( Frame *targetFrame, IMECHARPOSITION *charPos )
{
	if ( charPos->dwCharPos && !targetFrame->editor().hasComposition() )
	{
		return 0;
	}
	IntRect caret;
	if ( RefPtr<Range> range = targetFrame->editor().hasComposition() ? targetFrame->editor().compositionRange() : targetFrame->selection().selection().toNormalizedRange() )
	{
		RefPtr<Range> tempRange = range->cloneRange();
		tempRange->setStart( tempRange->startContainer(), tempRange->startOffset() + charPos->dwCharPos );
		caret = targetFrame->editor().firstRectForRange( tempRange.get() );
	}
	caret = targetFrame->view()->contentsToWindow( caret );
	caret.scale( deviceScaleFactor() );
	charPos->pt.x = caret.x();
	charPos->pt.y = caret.y();
	//::ClientToScreen( m_viewWindow, &charPos->pt );
	charPos->cLineHeight = caret.height();
	//::GetWindowRect( m_viewWindow, &charPos->rcDocument );
	return true;
}


LRESULT ViewIME::onIMERequestReconvertString( Frame *targetFrame, RECONVERTSTRING *reconvertString )
{
	RefPtr<Range> selectedRange = targetFrame->selection().toNormalizedRange();
	String text = selectedRange->text();
	if ( !reconvertString )
	{
		return sizeof( RECONVERTSTRING ) + text.length() * sizeof( UChar );
	}

	unsigned totalSize = sizeof( RECONVERTSTRING ) + text.length() * sizeof( UChar );
	if ( totalSize > reconvertString->dwSize )
	{
		return 0;
	}
	reconvertString->dwCompStrLen = text.length();
	reconvertString->dwStrLen = text.length();
	reconvertString->dwTargetStrLen = text.length();
	reconvertString->dwStrOffset = sizeof( RECONVERTSTRING );
	StringView( text ).getCharactersWithUpconvert( reinterpret_cast<UChar *>( reconvertString + 1 ) );
	return totalSize;
}


//	Win IME begin
bool ViewIME::onIMEStartComposition( WebView *wv )
{
	//LOG( TextInput, "onIMEStartComposition" );
	m_inIMEComposition++;
	Frame &targetFrame = wv->d->mPage->focusController().focusedOrMainFrame();

	HIMC hInputContext = getIMMContext();
	prepareCandidateWindow( &targetFrame, hInputContext );
	releaseIMMContext( hInputContext );
	return true;
}


bool ViewIME::onIMEComposition( WebView *wv, LPARAM lParam )
{
	//LOG( TextInput, "onIMEComposition %s", imeCompositionArgumentNames( lParam ).latin1().data() );
	HIMC hInputContext = getIMMContext();
	if ( !hInputContext )
	{
		return true;
	}

	Frame &targetFrame = wv->d->mPage->focusController().focusedOrMainFrame();
	if ( !targetFrame.editor().canEdit() )
	{
		return true;
	}

	prepareCandidateWindow( &targetFrame, hInputContext );

	if ( lParam & GCS_RESULTSTR || !lParam )
	{
		String compositionString;
		if ( !getCompositionString( hInputContext, GCS_RESULTSTR, compositionString ) && lParam )
		{
			return true;
		}

		targetFrame.editor().confirmComposition( compositionString );
	}
	else
	{
		String compositionString;
		if ( !getCompositionString( hInputContext, GCS_COMPSTR, compositionString ) )
		{
			return true;
		}

		// Composition string attributes
		int numAttributes = IMMDict::dict().getCompositionString( hInputContext, GCS_COMPATTR, 0, 0 );
		Vector<BYTE> attributes( numAttributes );
		IMMDict::dict().getCompositionString( hInputContext, GCS_COMPATTR, attributes.data(), numAttributes );

		// Get clauses
		int numClauses = IMMDict::dict().getCompositionString( hInputContext, GCS_COMPCLAUSE, 0, 0 );
		Vector<DWORD> clauses( numClauses / sizeof( DWORD ) );
		IMMDict::dict().getCompositionString( hInputContext, GCS_COMPCLAUSE, clauses.data(), numClauses );

		Vector<CompositionUnderline> underlines;
		compositionToUnderlines( clauses, attributes, underlines );

		int cursorPosition = LOWORD( IMMDict::dict().getCompositionString( hInputContext, GCS_CURSORPOS, 0, 0 ) );

		//const Editor& rEditor = targetFrame.editor();
		//rEditor.setComposition( compositionString, underlines, { }, cursorPosition, 0 );
		//targetFrame.editor().setComposition( compositionString, underlines, { }, cursorPosition, 0 );
		//targetFrame.editor().setComposition( compositionString, underlines, { }, cursorPosition, 0 );
	}

	return true;
}


bool ViewIME::onIMEEndComposition( WebView *wv )
{
	//LOG( TextInput, "onIMEEndComposition" );
	// If the composition hasn't been confirmed yet, it needs to be cancelled.
	// This happens after deleting the last character from inline input hole.
	Frame &targetFrame = wv->d->mPage->focusController().focusedOrMainFrame();
	if ( targetFrame.editor().hasComposition() )
	{
		targetFrame.editor().confirmComposition( String() );
	}

	if ( m_inIMEComposition )
	{
		m_inIMEComposition--;
	}

	return true;
}


bool ViewIME::onIMEChar( WebView *wv, WPARAM wParam, LPARAM lParam )
{
	UNUSED_PARAM( wParam );
	UNUSED_PARAM( lParam );
	//LOG( TextInput, "onIMEChar U+%04X %08X", wParam, lParam );
	return true;
}


bool ViewIME::onIMENotify( WebView *wv, WPARAM wParam, LPARAM lParam, LRESULT *pRlt )
{
	UNUSED_PARAM( wParam );
	//LOG( TextInput, "onIMENotify %s", imeNotificationName( wParam ).latin1().data() );
	return false;
}


LRESULT ViewIME::onIMERequest( WebView *wv, WPARAM wParam, LPARAM lParam )
{
	//LOG( TextInput, "onIMERequest %s", imeRequestName( request ).latin1().data() );
	Frame &targetFrame = wv->d->mPage->focusController().focusedOrMainFrame();
	if ( !targetFrame.editor().canEdit() )
	{
		return 0;
	}

	switch ( wParam )
	{
	case IMR_RECONVERTSTRING:
		return onIMERequestReconvertString( &targetFrame, (RECONVERTSTRING *)lParam );

	case IMR_QUERYCHARPOSITION:
		return onIMERequestCharPosition( &targetFrame, (IMECHARPOSITION *)lParam );
	}
	return 0;
}


bool ViewIME::onIMESelect( WebView *wv, WPARAM wParam, LPARAM lParam )
{
	UNUSED_PARAM( wParam );
	UNUSED_PARAM( lParam );
	//LOG( TextInput, "onIMESelect locale %ld %s", lParam, wParam ? "select" : "deselect" );
	return false;
}


bool ViewIME::onIMESetContext( WebView *wv, WPARAM wParam, LPARAM lParam )
{
	//LOG( TextInput, "onIMESetContext %s", wParam ? "active" : "inactive" );
	return false;
}
}; // namespace Telescope
#endif // #if TELESCOPE_USING( TELESCOPE_PC_PROGRAM )
