/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "AcceleratedCompositingContext.h"

#if USE( TEXTURE_MAPPER_TELESCOPE )

#include <WebCore/Document.h>
#include <WebCore/Frame.h>
#include <WebCore/FrameView.h>
#include <WebCore/GraphicsLayerTextureMapper.h>
#include <WebCore/Page.h>
#include <WebCore/Settings.h>
#include <WebCore/SystemInfo.h>
#include <WebCore/TextureMapperLayer.h>
#include <Webcore/TextureMapperTelescope.h>

#include "WebView.h"
#include "Universal/ViewPrivate.h"


using namespace WebCore;
namespace Telescope
{
AcceleratedCompositingContext::AcceleratedCompositingContext( WebView &webView, HardwareRenderBase &render )
	: m_webView( webView ), m_render( render )
{
}


void AcceleratedCompositingContext::initialize()
{
	if ( m_rootLayer )
	{
		return;
	}

	IntSize pageSize = m_webView.windowRect()->size();

	m_rootLayer = GraphicsLayer::create( nullptr, *this );
	m_rootLayer->setDrawsContent( false );
	m_rootLayer->setSize( pageSize );

	applyDeviceScaleFactor();

	m_nonCompositedContentLayer = GraphicsLayer::create( nullptr, *this );
	m_nonCompositedContentLayer->setDrawsContent( true );
	m_nonCompositedContentLayer->setContentsOpaque( false );
	m_nonCompositedContentLayer->setSize( pageSize );

	m_rootLayer->addChild( *m_nonCompositedContentLayer );
	m_nonCompositedContentLayer->setNeedsDisplay();

	m_textureMapper = TextureMapperTelescope::create( m_render );
	static_cast<TextureMapperTelescope *>( m_textureMapper )->setViewSize( pageSize );

	downcast<GraphicsLayerTextureMapper>( *m_rootLayer ).layer().setTextureMapper( m_textureMapper );

	flushPendingLayerChangesSoon();

#if ENABLE( DEVGUI )
	m_fpsTimestamp = WTF::MonotonicTime::now();
#endif // #if ENABLE( DEVGUI )
}

AcceleratedCompositingContext::~AcceleratedCompositingContext()
{
	if ( m_surface )
	{
		m_render.DeleteSurface( m_surface );
	}

#if ENABLE( DEVGUI )
	if ( m_pDevGUISurface )
	{
		m_render.DeleteSurface( m_pDevGUISurface );
	}
#endif // #if ENABLE( DEVGUI )
	delete m_textureMapper;
}


bool AcceleratedCompositingContext::enabled()
{
	return m_rootLayer && m_textureMapper;
}


bool AcceleratedCompositingContext::prepareForRendering()
{
	if ( !enabled() || !m_render.PrepareRendering() )
	{
		return false;
	}

	return true;
}


bool AcceleratedCompositingContext::startedAnimation( WebCore::GraphicsLayer *layer )
{
	if ( !layer )
	{
		return false;
	}

	return downcast<GraphicsLayerTextureMapper>( *layer ).layer().descendantsOrSelfHaveRunningAnimations();
}


void AcceleratedCompositingContext::applyDeviceScaleFactor()
{
	if ( !m_rootLayer )
	{
		return;
	}

	const FloatSize &size = m_rootLayer->size();

	WebCore::TransformationMatrix m;
	m.scale( deviceScaleFactor() );
	double tx = ( size.width() - size.width() / deviceScaleFactor() ) / 2.0;
	double ty = ( size.height() - size.height() / deviceScaleFactor() ) / 2.0;
	m.translate( tx, ty );
	m_rootLayer->setTransform( m );
}


void AcceleratedCompositingContext::updateRenderSurfaceIfNeeded()
{
	IntSize pageSize = m_webView.windowRect()->size();
	if ( !m_surface )
	{
		m_surface = m_render.CreateRenderTarget();
		if ( !m_surface )
		{
			return;
		}
		m_surface->SetDimension( pageSize.width(), pageSize.height() );
		static_cast<TextureMapperTelescope *>( m_textureMapper )->setViewSize( pageSize );
	}
	else if ( m_surface->Width() != pageSize.width() || m_surface->Height() != pageSize.height() )
	{
		m_surface->SetDimension( pageSize.width(), pageSize.height() );
		static_cast<TextureMapperTelescope *>( m_textureMapper )->setViewSize( pageSize );
	}

#if ENABLE( DEVGUI )
	if ( m_devGUIEnabled && !m_pDevGUISurface )
	{
		m_pDevGUISurface = m_render.CreateTexture();
		m_pDevGUISurface->SetDimension( DevGUIWidth, DevGUIHeight );
	}
	else if ( !m_devGUIEnabled && m_pDevGUISurface )
	{
		m_render.DeleteSurface( m_pDevGUISurface );
		m_pDevGUISurface = nullptr;
	}
#endif // #if ENABLE( DEVGUI )
}

bool AcceleratedCompositingContext::compositeLayersToContext()
{
	// If there are pending layer changes or has a ongoing animation
	// Flush the pending layer changes and do the composite operation
	if ( !m_pendingLayerChange && !startedAnimation( m_rootLayer.get() ) )
	{
		return false;
	}

	if ( !prepareForRendering() )
	{
		return false;
	}
	m_pendingLayerChange = false;
	m_render.BeginRendering();
	updateRenderSurfaceIfNeeded();
	downcast<GraphicsLayerTextureMapper>( *m_rootLayer ).updateBackingStoreIncludingSubLayers();
	m_render.SetRenderTarget( *m_surface );
	m_render.ClearColor( TSColor() );
	m_textureMapper->beginPainting();
	downcast<GraphicsLayerTextureMapper>( *m_rootLayer ).layer().paint();

#if ENABLE( DEVGUI )
	UpdateFPS();
	if ( m_devGUIEnabled )
	{
		renderDevGUI();
	}
#endif // #if ENABLE( DEVGUI )

	m_textureMapper->endPainting();
	m_render.EndRendering();
	return true;
}


void AcceleratedCompositingContext::setRootCompositingLayer( GraphicsLayer *graphicsLayer )
{
	m_targetGraphicsLayer = graphicsLayer;
	m_targetGraphicsLayerDirty = true;
	flushPendingLayerChangesSoon();
}


void AcceleratedCompositingContext::doSetRootCompositingLayer( GraphicsLayer *graphicsLayer )
{
	if ( !graphicsLayer )
	{
		m_rootLayer = nullptr;
		m_nonCompositedContentLayer = nullptr;
		if ( m_textureMapper )
		{
			delete m_textureMapper;
			m_textureMapper = nullptr;
		}

		if ( m_surface )
		{
			m_render.DeleteSurface( m_surface );
			m_surface = nullptr;
		}
		return;
	}

	initialize();

	m_nonCompositedContentLayer->removeAllChildren();
	m_nonCompositedContentLayer->addChild( *graphicsLayer );
}


void AcceleratedCompositingContext::setNonCompositedContentsNeedDisplay( const IntRect &rect )
{
	if ( !m_rootLayer )
	{
		return;
	}

	if ( rect.isEmpty() )
	{
		m_rootLayer->setNeedsDisplay();
		return;
	}
	m_nonCompositedContentLayer->setNeedsDisplayInRect( rect );
	flushPendingLayerChangesSoon();
}


void AcceleratedCompositingContext::resizeRootLayer( const IntSize &newSize )
{
	if ( !enabled() || m_rootLayer->size() == newSize )
	{
		return;
	}

	m_rootLayer->setSize( newSize );

	applyDeviceScaleFactor();

	FloatSize oldSize = m_nonCompositedContentLayer->size();
	m_nonCompositedContentLayer->setSize( newSize );

	if ( newSize.width() > oldSize.width() )
	{
		float height = std::min( static_cast<float>( newSize.height() ), oldSize.height() );
		m_nonCompositedContentLayer->setNeedsDisplayInRect( FloatRect( oldSize.width(), 0, newSize.width() - oldSize.width(), height ) );
	}

	if ( newSize.height() > oldSize.height() )
	{
		m_nonCompositedContentLayer->setNeedsDisplayInRect( FloatRect( 0, oldSize.height(), newSize.width(), newSize.height() - oldSize.height() ) );
	}

	m_nonCompositedContentLayer->setNeedsDisplayInRect( IntRect( IntPoint(), newSize ) );
	flushPendingLayerChangesSoon();
}


void AcceleratedCompositingContext::scrollNonCompositedContents( const IntRect &scrollRect, const IntSize & /* scrollOffset */ )
{
	m_nonCompositedContentLayer->setNeedsDisplay();
	flushPendingLayerChangesSoon();
}


const HardwareSurfaceBase *AcceleratedCompositingContext::getRenderSurface()
{
	return enabled() ? m_surface : nullptr;
}


bool AcceleratedCompositingContext::flushPendingLayerChanges()
{
	FrameView *frameView = m_webView.TopLevelFrame()->view();
	m_rootLayer->flushCompositingStateForThisLayerOnly();
	m_nonCompositedContentLayer->flushCompositingStateForThisLayerOnly();
	if ( !frameView->flushCompositingStateIncludingSubframes() )
	{
		return false;
	}

	return true;
}


bool AcceleratedCompositingContext::flushAndRenderLayers()
{
	if ( m_targetGraphicsLayerDirty )
	{
		doSetRootCompositingLayer( m_targetGraphicsLayer );
		m_targetGraphicsLayerDirty = false;
	}

	if ( !enabled() )
	{
		return false;
	}

	Frame &frame = *m_webView.TopLevelFrame();
	if ( !frame.contentRenderer() || !frame.view() )
	{
		return false;
	}

	if ( !flushPendingLayerChanges() && !m_pendingLayerChange )
	{
		return false;
	}

	return compositeLayersToContext();
}


void AcceleratedCompositingContext::paintContents( const GraphicsLayer *, GraphicsContext &context, const FloatRect &rectToPaint, GraphicsLayerPaintBehavior )
{
	context.save();
	context.clip( rectToPaint );
	m_webView.TopLevelFrame()->view()->paint( context, enclosingIntRect( rectToPaint ) );
	context.restore();
}


float AcceleratedCompositingContext::deviceScaleFactor() const
{
	return 1.0f;
}


void AcceleratedCompositingContext::flushPendingLayerChangesSoon()
{
	if ( enabled() )
	{
		m_pendingLayerChange = true;
	}
}


void AcceleratedCompositingContext::notifyFlushRequired( const WebCore::GraphicsLayer * )
{
	flushPendingLayerChangesSoon();
}

#if ENABLE( DEVGUI )
void AcceleratedCompositingContext::UpdateFPS()
{
	m_fpsframeCounter++;
	Seconds delta = MonotonicTime::now() - m_fpsTimestamp;
	if ( delta >= WTF::Seconds( 1.0f ) )
	{
		m_lastFPS = int( m_fpsframeCounter / delta.seconds() );
		m_fpsframeCounter = 0;
		m_fpsTimestamp += delta;
	}
	m_frameIndex++;
}


void AcceleratedCompositingContext::DrawDebugInfo( cairo_t *cairoContext, DevGUIContext &guiContext )
{
	guiContext.currentY += guiContext.fontSize;
	cairo_move_to( cairoContext, 0, guiContext.currentY );
	char debugInfo[255] = { 0 };
	sprintf_s( debugInfo, "Telescope FPS: %d", m_lastFPS );
	cairo_show_text( cairoContext, debugInfo );

	guiContext.currentY += guiContext.fontSize;
	cairo_move_to( cairoContext, 0, guiContext.currentY );
	sprintf_s( debugInfo, "Frame index: %d", m_frameIndex );
	cairo_show_text( cairoContext, debugInfo );

	guiContext.currentY += guiContext.fontSize;
	cairo_move_to( cairoContext, 0, guiContext.currentY );
	sprintf_s( debugInfo, "Rendering size: %dx%d", m_surface->Width(), m_surface->Height() );
	cairo_show_text( cairoContext, debugInfo );
}


void AcceleratedCompositingContext::renderDevGUI()
{
	if ( !m_pDevGUISurface )
	{
		return;
	}

	if ( m_render.PrepareTex2DSubData( Telescope::TSIntRect( 0, 0, DevGUIWidth, DevGUIHeight ) ) )
	{
		float defaultFontSize = 25;
		cairo_surface_t *surface = cairo_image_surface_create( CAIRO_FORMAT_ARGB32, DevGUIWidth, DevGUIHeight );
		cairo_t *cr = cairo_create( surface );

		// clear color
		cairo_set_source_rgba( cr, 0, 0, 0, 0 );
		cairo_rectangle( cr, 0, 0, DevGUIWidth, DevGUIHeight );
		cairo_fill( cr );

		// setup font face, size and color
		cairo_select_font_face( cr, "Monospace", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD );
		cairo_set_font_size( cr, defaultFontSize );
		cairo_set_source_rgb( cr, 1, 0, 0 );

		// render devgui
		DevGUIContext context = { 0, defaultFontSize, DevGUIWidth, DevGUIHeight };

		DrawDevGui( cr, context );

		// Update devgui texture
		TSIntRect sourceRect( 0, 0, DevGUIWidth, DevGUIHeight );
		const unsigned char *bits = cairo_image_surface_get_data( surface );
		int stride = cairo_image_surface_get_stride( surface );
		m_render.Tex2DSubData( *m_pDevGUISurface, bits, sourceRect, TSIntPoint( 0, 0 ), stride );
		cairo_surface_destroy( surface );
		cairo_destroy( cr );
	}

	// draw devgui to render target
	float scaleFactor = 1.0f;
	if ( m_surface->Width() > 1920 )
	{
		// for large view scale 2 times
		scaleFactor = 2.0f;
	}
	m_render.DrawSurface( *m_pDevGUISurface, TSFloatRect( 0, 0, DevGUIWidth, DevGUIHeight ), TSFloatPoint( 5, 0 ), TSFloatSize( scaleFactor, scaleFactor ), 0, 1.0f );
}
#endif // #if ENABLE( DEVGUI )
} // namespace Telescope
#endif // #if USE( TEXTURE_MAPPER_TELESCOPE )
