/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "WebFrame.h"
#include "WebView.h"
#include "WebFrameLoaderClient.h"
#include "webcore/Frame.h"
#include "webcore/Page.h"
#include "WebCore/FrameLoadRequest.h"
#include "webcore/ResourceRequest.h"
#include "webcore/FrameLoader.h"
#include "WebCore/JSDOMWindowBase.h"
#include "WebCore/DOMWrapperWorld.h"
#include "WebCore/ScriptController.h"
#include "JavaScriptCore/APICast.h"
#include "JavaScriptCore/JSGlobalObject.h"
#include "JavaScriptCore/PropertyName.h"
#include "JavaScript/TelescopeInstance.h"
#include "MemoryTrackingScope.h"

using namespace WebCore;
using namespace Telescope;

#define JSBINDING_BASIC_CHECK()                                                                \
	WebCore::JSDOMWindow *pWindow = toJSDOMWindow( mFrame, WebCore::mainThreadNormalWorld() ); \
	JSC::Bindings::RootObject *pRoot = mFrame->script().bindingRootObject();                   \
	if ( !pWindow || !pRoot )                                                                  \
	{                                                                                          \
		assert( 0 );                                                                           \
	}                                                                                          \
	JSC_GlobalObject *pGlobalObj = pWindow->globalObject();                                    \
	JSC::JSLockHolder lock( pGlobalObj );

WebCore::Frame *core( WebFrame *webFrame )
{
	return webFrame->frameView();
}


WebFrame::WebFrame()
	: mFrame( nullptr )
	, mWebView( nullptr )
{
}


WebFrame::~WebFrame()
{
}


Ref<WebCore::Frame> WebFrame::createSubframeWithOwnerElement( WebView *view, WebCore::Page *page, WebCore::HTMLFrameOwnerElement *ownerElement )
{
	mWebView = view;
	auto frame = Frame::create( page, ownerElement, new WebFrameLoaderClient( this ) );
	mFrame = frame.ptr();
	return frame;
}


void WebFrame::initWithWebView( ::Telescope::WebView *view, WebCore::Page *page )
{
	mWebView = view;
	mFrame = &page->mainFrame();
}


WebView *WebFrame::webView() const
{
	return mWebView;
}


WebFrame *WebFrame::createInstance()
{
	WebFrame *instance = new WebFrame();
	return instance;
}


void WebFrame::loadRequest( WTF::URL *url )
{
	TELESCOPE_MEMORY_SCOPE( "WebFrame::loadRequest" );

	Frame *coreFrame = core( this );
	if ( !coreFrame )
	{
		return;
	}
	coreFrame->loader().load( FrameLoadRequest( *coreFrame, ResourceRequest( *url ), WebCore::ShouldOpenExternalURLsPolicy::ShouldAllow ) );
}


void WebFrame::AddToJavaScriptWindowObject( const char *pName, Telescope::IJSBoundObject *pObj )
{
	JSBINDING_BASIC_CHECK();

	JSC_Object *pRuntimeObj = JSC::Bindings::TelescopeInstance::Create( pObj, pRoot )->createRuntimeObject( pGlobalObj );
	JSC::Identifier oIdt = JSC::Identifier::fromString( pGlobalObj->vm(), pName );

	JSC::PutPropertySlot oSlot( pWindow );

	JSC::PropertyName oPropertyName( oIdt );
	pWindow->methodTable( pGlobalObj->vm() )->put( pWindow, pGlobalObj, oPropertyName, pRuntimeObj, oSlot );
}


Telescope::TelescopeJSValue *WebFrame::CreateJavaScriptValue()
{
	JSBINDING_BASIC_CHECK();

	JSC_Value newValue = JSC::jsUndefined();
	TelescopeJSValue *v = new TelescopeJSValue( &newValue, pGlobalObj );
	return v;
}


void WebFrame::DestroyJavaScriptValue( Telescope::TelescopeJSValue *value )
{
	JSBINDING_BASIC_CHECK();

	delete value;
}
