/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#pragma once

namespace Telescope
{
class MediaPlayerWrapperBase
{
public:
	struct RGBBuffer
	{
		RGBBuffer()
			: pRGBBuffer( nullptr )
			, bufferRowPitch( 0 )
			, bufferWidth( 0 )
			, bufferHeight( 0 )
		{
		}

		bool IsValid() const
		{
			return pRGBBuffer && bufferRowPitch > 0 && bufferWidth > 0 && bufferHeight > 0;
		}

		const void *pRGBBuffer;
		unsigned int bufferRowPitch;
		unsigned int bufferWidth;
		unsigned int bufferHeight;
	};

	virtual void Load( const char *pUrl ) = 0;
	virtual RGBBuffer GetVideoRGBBuffer() = 0;
	virtual void Update() = 0;
	virtual void RenderTick() = 0;

	virtual bool IsAvailable() const = 0;
	virtual float CurrentTime() const = 0;
	virtual float Duration() const = 0;

	virtual void Play( bool play ) = 0;
	virtual void Mute( bool mute ) = 0;
	virtual void Loop( bool loop ) = 0;
	virtual void Seek( float seconds ) = 0;

	virtual bool HasVideo() const = 0;
	virtual bool HasAudio() const = 0;

	virtual void GetVideoSize( size_t &width, size_t &height ) const = 0;

	virtual void GetDebugInfo( int &renderedFrames, int &droppedFrames ) const = 0;

	virtual bool AcceleratedRenderingEnabled() const = 0;

	virtual ~MediaPlayerWrapperBase()
	{
	}
};

class MediaPlayerWrapperEventHandler
{
public:
	virtual void OnMetadataReady() = 0;
	virtual void OnReadyPlay() = 0;
	virtual void OnEndPlay() = 0;
	virtual void OnFatalError() = 0;

	virtual void OnNewVideoFrame() = 0;

protected:
	virtual ~MediaPlayerWrapperEventHandler()
	{
	}
};

class MediaPlayerWrapperClientFactoryBase
{
public:
	virtual MediaPlayerWrapperBase *CreateMediaPlayerWrapper( MediaPlayerWrapperEventHandler &pEventHandler ) = 0;
	virtual void DestroyMediaPlayerWrapper( MediaPlayerWrapperBase *pMediaPlayerWrapper ) = 0;

protected:
	virtual ~MediaPlayerWrapperClientFactoryBase()
	{
	}
};
} // namespace Telescope