/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include <stdlib.h> // for size_t
#include "MemoryTag.h"

namespace Telescope
{
class IBasicMemoryAllocator
{
public:
	IBasicMemoryAllocator() {}
	virtual ~IBasicMemoryAllocator() {}

	virtual void *Allocate( size_t size, const MemoryTag &tag ) = 0;
	virtual void Deallocate( void *p ) = 0;
	virtual void *TryAllocate( size_t size, const MemoryTag &tag )
	{
		return Allocate( size, tag );
	}

private:
	IBasicMemoryAllocator( const IBasicMemoryAllocator & ) = delete;
	IBasicMemoryAllocator( const IBasicMemoryAllocator && ) = delete;
	void operator=( const IBasicMemoryAllocator & ) = delete;
	void operator=( const IBasicMemoryAllocator && ) = delete;
};

class IMemoryAllocator : public IBasicMemoryAllocator
{
public:
	virtual void *AllocateAligned( size_t size, size_t align, const MemoryTag &tag ) = 0;
	virtual void *Reallocate( void *p, size_t size, const MemoryTag &tag ) = 0;
	virtual void DeallocateAligned( void *p ) = 0;
	virtual size_t AllocateSize( void *p ) = 0;
	virtual void *TryAllocateAligned( size_t size, size_t align, const MemoryTag &tag )
	{
		return AllocateAligned( size, align, tag );
	}
	virtual void *TryReallocate( void *p, size_t size, const MemoryTag &tag )
	{
		return Reallocate( p, size, tag );
	}
};

void InstallMemoryAllocator( IMemoryAllocator *allocator = nullptr );
void UninstallMemoryAllocator();

inline IMemoryAllocator *Telescope_GetMemoryAllocator()
{
	extern IMemoryAllocator *g_pMemoryAllocator;
	return g_pMemoryAllocator;
}
} // namespace Telescope