/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#pragma once

#include "PlatformFeatureDefs.h"

// As we need to keep the binary compatibility of Telescope.dll among build configurations.
// We must keep the following classes always working and having the same binary representation
// regardless of which build configuration is used either by Telescope project or host projects.
//
// So the following classes are always defined regardless of macro MEMORY_TRACING is on or not.

namespace Telescope
{
//
// IMemoryScopeTracker
//

class IMemoryScopeTracker
{
public:
	// The following 3 methods are related with the current thread (the calling thread)
	virtual void *PushScope( const char *pName ) = 0;
	virtual void PopScope( void *pContext ) = 0;
	virtual void GetScopeStack( const void *&pOutStack, size_t &outCount ) const = 0;
	virtual const char *GetScope( const void *pStack, size_t index ) const = 0;

protected:
	virtual ~IMemoryScopeTracker() {}
};


//
// MemoryTrackingScope
//

class MemoryTrackingScope
{
	IMemoryScopeTracker *m_pTracker;
	void *m_pContext;

public:
#if TELESCOPE_USING( TELESCOPE_MEMORY_TRACING )
	MemoryTrackingScope( IMemoryScopeTracker *pTracker, const char *pName )
		: m_pTracker( pTracker )
		, m_pContext( pTracker ? pTracker->PushScope( pName ) : nullptr )
	{
	}

	~MemoryTrackingScope()
	{
		if ( m_pTracker )
		{
			m_pTracker->PopScope( m_pContext );
		}
	}
#else // #if TELESCOPE_USING( TELESCOPE_MEMORY_TRACING )
	MemoryTrackingScope( IMemoryScopeTracker &, const char * )
		: m_pTracker( nullptr )
		, m_pContext( nullptr )
	{
	}

	~MemoryTrackingScope()
	{
		// Let PlayStation compiler shutup for the warning of 'unused data members'
		m_pTracker = nullptr;
		m_pContext = nullptr;
	}
#endif // #else // #if TELESCOPE_USING( TELESCOPE_MEMORY_TRACING )
};
} // namespace Telescope


//
// Scope macros
//

#include "TelescopeLib.h"

#if TELESCOPE_USING( TELESCOPE_MEMORY_TRACING )

// This is some crazy magic that helps produce unique identifiers in C++ code.
// I still can't figure out why it works.
#define TELESCOPE_UNIQUE_ID_CAT0( a, b )   TELESCOPE_UNIQUE_ID_CAT1( a, b )
#define TELESCOPE_UNIQUE_ID_CAT1( a, b )   TELESCOPE_UNIQUE_ID_CAT2( ~, a##b )
#define TELESCOPE_UNIQUE_ID_CAT2( p, res ) res
#define TELESCOPE_UNIQUE_ID( base )		   TELESCOPE_UNIQUE_ID_CAT0( base, __COUNTER__ )

#include "TelescopeDll.h"

#define TELESCOPE_MEMORY_SCOPE( name ) \
	Telescope::MemoryTrackingScope     \
		TELESCOPE_UNIQUE_ID( __memoryTrackerScope_ )( &CreateTelescopeInstance()->GetMemoryScopeTracker(), name )

#else // #if TELESCOPE_USING( TELESCOPE_MEMORY_TRACING )

#define TELESCOPE_MEMORY_SCOPE( name )

#endif // #else // #if TELESCOPE_USING( TELESCOPE_MEMORY_TRACING )