/*
 * Copyright (C) 2013, 2016 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include <wtf/Noncopyable.h>
#include <wtf/Vector.h>

namespace PAL {

struct CryptoDigestContext;

class CryptoDigest {
    WTF_MAKE_NONCOPYABLE(CryptoDigest);
    WTF_MAKE_FAST_ALLOCATED; /*TELESCOPE_LIB_CHANGES 12/Apr/2020 Customize PAL memory allocation*/
public:
    enum class Algorithm {
        SHA_1,
        SHA_224,
        SHA_256,
        SHA_384,
        SHA_512,
    };
    PAL_EXPORT static std::unique_ptr<CryptoDigest> create(Algorithm);
    PAL_EXPORT ~CryptoDigest();

    PAL_EXPORT void addBytes(const void* input, size_t length);
    PAL_EXPORT Vector<uint8_t> computeHash();

private:
    CryptoDigest();

    std::unique_ptr<CryptoDigestContext> m_context;
};

} // namespace PAL

//+TELESCOPE_LIB_CHANGES
// 27/May/2020
// Conform to an uniform interface for Telescope
#define SHA_DIGEST_LENGTH       20
#define SHA224_DIGEST_LENGTH	28
#define SHA256_DIGEST_LENGTH	32
#define SHA384_DIGEST_LENGTH	48
#define SHA512_DIGEST_LENGTH	64

void* SHA1_Create();
int   SHA1_Init( void* context );
int   SHA1_Update( void* context, const void* data, size_t len );
int   SHA1_Final( unsigned char* md, void* context );
void  SHA1_Destroy( void* context );

void* SHA224_Create();
int   SHA224_Init( void* context );
int   SHA224_Update( void* context, const void* data, size_t len );
int   SHA224_Final( unsigned char* md, void* context );
void  SHA224_Destroy( void* context );

void* SHA256_Create();
int   SHA256_Init( void* context );
int   SHA256_Update( void* context, const void* data, size_t len );
int   SHA256_Final( unsigned char* md, void* context );
void  SHA256_Destroy( void* context );

void* SHA384_Create();
int   SHA384_Init( void* context );
int   SHA384_Update( void* context, const void* data, size_t len );
int   SHA384_Final( unsigned char* md, void* context );
void  SHA384_Destroy( void* context );

void* SHA512_Create();
int   SHA512_Init( void* context );
int   SHA512_Update( void* context, const void* data, size_t len );
int   SHA512_Final( unsigned char* md, void* context );
void  SHA512_Destroy( void* context );
//-TELESCOPE_LIB_CHANGES
