/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#if USE(TEXTURE_MAPPER_TELESCOPE)
#include <wtf/FastMalloc.h>
#include "BitmapTextureTelescope.h"
#include <cairo.h>
#include "Image.h"
#include "Logging.h"
#include "NativeImage.h"

namespace WebCore
{
Ref<BitmapTexture> BitmapTextureTelescope::createTexture(Telescope::HardwareRenderBase& render)
{
	Telescope::HardwareSurfaceBase* surface = render.CreateTexture();
	if (!surface)
	{
		// This may happen when host have limited hardware surface resource and won't able to
		// allocate more texture, we would tolerate this error
		RELEASE_LOG_ERROR(Compositing, "Failed Create new surface");
	}
	return adoptRef(*new BitmapTextureTelescope(render, surface));
}

Ref<BitmapTexture> BitmapTextureTelescope::createRenderTarget( Telescope::HardwareRenderBase& render )
{
	Telescope::HardwareSurfaceBase* surface = render.CreateRenderTarget();
	if ( !surface )
	{
		// This may happen when host have limited hardware surface resource and won't able to
		// allocate more texture, we would tolerate this error
		RELEASE_LOG_ERROR( Compositing, "Failed Create new surface" );
	}
	return adoptRef( *new BitmapTextureTelescope( render, surface ) );
}

WebCore::IntSize BitmapTextureTelescope::size() const
{
	assert(m_surface);
	if (m_surface)
	{
		return IntSize(m_surface->Width(), m_surface->Height());
	}
	else
	{
		return IntSize(0, 0);
	}
}

bool BitmapTextureTelescope::isValid() const
{
	return m_surface != nullptr && m_surface->IsAllocated() && !m_contentSize.isEmpty();
}

void BitmapTextureTelescope::updateContents(Image* image, const IntRect& targetRect, const IntPoint& offset)
{
	if (!isValid() || !image)
	{
		return;
	}

	NativeImagePtr frameImage = image->nativeImageForCurrentFrame();
	if (!frameImage)
	{
		return;
	}

	int bytesPerLine;
	const char* imageData;

#if USE(CAIRO)
	cairo_surface_t* surface = frameImage.get();
	imageData = reinterpret_cast<const char*>(cairo_image_surface_get_data(surface));
	bytesPerLine = cairo_image_surface_get_stride(surface);
#endif

	updateContents(imageData, targetRect, offset, bytesPerLine);
}

void BitmapTextureTelescope::updateContents(const void* data, const IntRect& target, const IntPoint& offset, int bytesPerLine)
{
	if (!isValid())
	{
		return;
	}

	m_render.Tex2DSubData(*m_surface, (const char*)data, target, offset, bytesPerLine );
}

void BitmapTextureTelescope::didReset()
{
	m_shouldClear = true;

	if (m_surface)
	{
		m_surface->SetDimension(m_contentSize.width(), m_contentSize.height());
	}
}

RefPtr<WebCore::BitmapTexture> BitmapTextureTelescope::applyFilters(TextureMapper&, const FilterOperations& filters)
{
	if (!m_filters)
	{
		m_filters = std::make_unique<FilterOperations>();
	}

	*m_filters = filters;
	return this;
}

void BitmapTextureTelescope::initializeStencil()
{
	if (m_clearedStencil)
	{
		return;
	}

	m_render.ClearStencil(0);
	m_render.ClearStencil(0);
	m_clearedStencil = true;
}

void BitmapTextureTelescope::bind()
{
	if (!isValid())
	{
		return;
	}

	m_render.SetRenderTarget(*getSurface());
	if (m_shouldClear)
	{
		m_clipstack.reset(IntRect(IntPoint::zero(), m_contentSize), ClipStack::YAxisMode::Default);
		m_clipstack.applyIfNeeded();
		m_render.ClearColor(Telescope::TSColor());
		m_shouldClear = false;
	}
	m_clipstack.apply();
}

BitmapTextureTelescope::BitmapTextureTelescope(Telescope::HardwareRenderBase& render, Telescope::HardwareSurfaceBase* surface)
	: m_render(render), m_surface(surface), m_clipstack(render)
{
}

BitmapTextureTelescope::~BitmapTextureTelescope()
{
	if (m_surface)
	{
		m_render.DeleteSurface(m_surface);
		m_surface = nullptr;
	}
}

const FilterOperations* BitmapTextureTelescope::getFilters() const
{
	return m_filters.get();
}

}  // namespace WebCore

#endif	// #if USE(TEXTURE_MAPPER_TELESCOPE)