/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "config.h"
#if USE(TEXTURE_MAPPER_TELESCOPE)
#include <wtf/FastMalloc.h>
#include "TextureMapperTelescope.h"
#include "BitmapTexturePool.h"
#include "BitmapTextureTelescope.h"
#include "FloatQuad.h"
#include "NotImplemented.h"

#define USE_WEBCORE_TYPE_CONVERTION_UTILITY
#include <Telescope/include/Render/HardwareRenderBase.h>
#undef USE_WEBCORE_TYPE_CONVERTION_UTILITY

#if USE(CAIRO)
#include <cairo.h>
#include <wtf/text/CString.h>
#include "CairoUtilities.h"
#include "RefPtrCairo.h"
#endif

namespace WebCore
{
TextureMapperTelescope* TextureMapperTelescope::create(Telescope::HardwareRenderBase& render)
{
	return new TextureMapperTelescope(render);
}

TextureMapperTelescope::TextureMapperTelescope(Telescope::HardwareRenderBase& render) 
	: m_render(render)
	, m_clipStack(render)
{
	m_texturePool = makeUnique<BitmapTexturePool>(m_render);
}

TextureMapperTelescope::~TextureMapperTelescope() {}

void TextureMapperTelescope::drawBorder(const Color& color, float borderWidth, const FloatRect& rect, const TransformationMatrix& transform)
{
	if (clipStack().isCurrentScissorBoxEmpty())
		return;
	TransformationMatrix::Decomposed2Type decomposed;
	transform.decompose2(decomposed);
	m_render.DrawBorder(color, borderWidth, rect, Telescope::TSFloatPoint(decomposed.translateX, decomposed.translateY)
		,Telescope::TSFloatSize(decomposed.scaleX, decomposed.scaleY), deg2rad(decomposed.angle) );
}

void TextureMapperTelescope::drawNumber(int number, const Color& color, const FloatPoint& targetPoint, const TransformationMatrix& modelViewMatrix)
{
	const int pointSize = 8;

#if USE(CAIRO)
	CString counterString = String::number(number).ascii();
	// cairo_text_extents() requires a cairo_t, so dimensions need to be guesstimated.
	int width = static_cast<int>(counterString.length() * pointSize * 1.2);
	int height = static_cast<int>(pointSize * 1.5);

	cairo_surface_t* surface = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, width, height);
	cairo_t* cr = cairo_create(surface);

	// Since we won't swap R+B when uploading a texture, paint with the swapped R+B color.
	if (color.isExtended())
		cairo_set_source_rgba(cr, color.asExtended().blue(), color.asExtended().green(), color.asExtended().red(), color.asExtended().alpha());
	else
	{
		float r, g, b, a;
		color.getRGBA(r, g, b, a);
		cairo_set_source_rgba(cr, b, g, r, a);
	}

	cairo_rectangle(cr, 0, 0, width, height);
	cairo_fill(cr);

	cairo_select_font_face(cr, "Monospace", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
	cairo_set_font_size(cr, pointSize);
	cairo_set_source_rgb(cr, 1, 1, 1);
	cairo_move_to(cr, 2, pointSize);
	cairo_show_text(cr, counterString.data());

	IntSize size(width, height);
	IntRect sourceRect(IntPoint::zero(), size);
	IntRect targetRect(roundedIntPoint(targetPoint), size);

	RefPtr<BitmapTexture> texture = acquireTextureFromPool(size);
	const unsigned char* bits = cairo_image_surface_get_data(surface);
	int stride = cairo_image_surface_get_stride(surface);
	static_cast<BitmapTextureTelescope*>(texture.get())->updateContents(bits, sourceRect, IntPoint::zero(), stride);
	drawTexture(*texture, targetRect, modelViewMatrix, 1.0f, AllEdges);

	cairo_surface_destroy(surface);
	cairo_destroy(cr);

#else
	UNUSED_PARAM(number);
	UNUSED_PARAM(pointSize);
	UNUSED_PARAM(targetPoint);
	UNUSED_PARAM(modelViewMatrix);
	notImplemented();
#endif
}

void TextureMapperTelescope::drawTexture(const BitmapTexture& texture,
										 const FloatRect& targetRect,
										 const TransformationMatrix& matrix /*= TransformationMatrix()*/,
										 float opacity /*= 1.0f*/,
										 unsigned exposedEdges /*= AllEdges */)
{
	if (!texture.isValid())
	{
		return;
	}

	if (clipStack().isCurrentScissorBoxEmpty())
	{
		return;
	}

	TransformationMatrix::Decomposed2Type decomposed;
	matrix.decompose2( decomposed );

	const BitmapTextureTelescope& textureTelescope = static_cast<const BitmapTextureTelescope&>(texture);
	m_render.DrawSurface( *textureTelescope.getSurface(), targetRect, Telescope::TSFloatPoint( decomposed.translateX, decomposed.translateY )
		, Telescope::TSFloatSize(decomposed.scaleX, decomposed.scaleY), deg2rad(decomposed.angle), opacity);
}

void TextureMapperTelescope::drawSolidColor(const FloatRect& targetRect, const TransformationMatrix& matrix, const Color& color, bool)
{
	m_render.Fill(color , targetRect, Telescope::TSFloatPoint( matrix.m41(), matrix.m42()));
}

void TextureMapperTelescope::clearColor(const Color& color)
{
	m_render.ClearColor(color);
}

void TextureMapperTelescope::bindDefaultSurface()
{
	ASSERT(m_defaultSurface);
	m_currentSurface = nullptr;
	m_render.SetRenderTarget(*m_defaultSurface);
	m_clipStack.apply();
}

void TextureMapperTelescope::bindSurface(BitmapTexture* inSurface)
{
	if (!inSurface)
	{
		bindDefaultSurface();
	}
	else
	{
		static_cast<BitmapTextureTelescope*>(inSurface)->bind();
		m_currentSurface = inSurface;
	}
}

bool TextureMapperTelescope::beginScissorClip(const TransformationMatrix& transform, const FloatRect& targetRect)
{
	if (!transform.isAffine())
	{
		return false;
	}

	FloatQuad quad = transform.projectQuad(targetRect);
	IntRect rect = quad.enclosingBoundingBox();
	if (!quad.isRectilinear() || rect.isEmpty())
	{
		return false;
	}

	clipStack().intersect(rect);
	clipStack().applyIfNeeded();
	return true;
}

void TextureMapperTelescope::initializeStencil()
{
	if (m_currentSurface)
	{
		static_cast<BitmapTextureTelescope*>(m_currentSurface.get())->initializeStencil();
		return;
	}

	if (m_didModifyStencil)
	{
		return;
	}

	m_render.ClearStencil(0);
	m_didModifyStencil = true;
}

void TextureMapperTelescope::beginClip(const TransformationMatrix& transform, const FloatRect& rect)
{
	clipStack().push();
	if (beginScissorClip(transform, rect))
	{
		return;
	}

	initializeStencil();

	int stencilIndex = clipStack().getStencilIndex();
	TransformationMatrix::Decomposed2Type decomposed;
	transform.decompose2( decomposed );
	m_render.DrawStencil(stencilIndex, rect, Telescope::TSFloatPoint( decomposed.translateX, decomposed.translateY ), 
		Telescope::TSFloatSize(decomposed.scaleX, decomposed.scaleY), deg2rad(decomposed.angle));

	clipStack().setStencilIndex(stencilIndex << 1);
	clipStack().applyIfNeeded();
}

void TextureMapperTelescope::endClip()
{
	clipStack().pop();
	clipStack().applyIfNeeded();
}

WebCore::IntRect TextureMapperTelescope::clipBounds()
{
	return clipStack().current().scissorBox;
}

WebCore::ClipStack& TextureMapperTelescope::clipStack()
{
	return m_currentSurface ? static_cast<BitmapTextureTelescope*>(m_currentSurface.get())->clipStack() : m_clipStack;
}

void TextureMapperTelescope::beginPainting(PaintFlags /* = 0 */)
{
	m_didModifyStencil = false;
	m_defaultSurface = m_render.GetRenderTarget();
	m_clipStack.reset(IntRect(IntPoint::zero(), m_viewSize), ClipStack::YAxisMode::Default);
}

void TextureMapperTelescope::endPainting()
{
	if (m_didModifyStencil)
	{
		m_render.ClearStencil(1);
	}
}

Ref<WebCore::BitmapTexture> TextureMapperTelescope::createTexture()
{
	return BitmapTextureTelescope::createTexture(m_render);
}

Ref<WebCore::BitmapTexture> TextureMapperTelescope::createTexture(int internalFormat)
{
	UNUSED_VAR(internalFormat);
	return createTexture();
}

Ref<WebCore::BitmapTexture> TextureMapperTelescope::createRenderTarget()
{
	return BitmapTextureTelescope::createRenderTarget(m_render);
}

void TextureMapperTelescope::setViewSize(const IntSize& viewSize)
{
	m_viewSize = viewSize;
}

}  // namespace WebCore
#endif	// #if USE(TEXTURE_MAPPER_TELESCOPE)