/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following 
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products  
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 

#include "config.h"
#include "TelescopeWebKitThreadInterface.h"
#include <wtf/MainThread.h>
#include <../WebCore/platform/SharedTimer.h>
#include <cassert>

namespace Telescope
{
namespace WebKit
{
class DefaultMutex : public IMutex
{
public:
	friend class DefaultThreadCondition;
	DefaultMutex()
		: mLockCount(0)
	{

	}
	void Lock() 
	{
		++mLockCount;
	}

	bool TryLock()
	{
		if(mLockCount)
			return false;
		
		Lock();
		return true;
	}

	void Unlock()
	{
		--mLockCount;
	}
private:
	int mLockCount;
};

class DefaultThreadCondition : public IThreadCondition
{
public:
	void Wait(IMutex* mutex)
	{
		
	}

	bool TimedWait(IMutex* mutex, double relativeTimeMS)
	{
		return false;
	}

	void Signal(bool broadcast)
	{

	}
private:
};


class DefaultThread : public IThread
{
public:
	virtual ThreadId Begin(ThreadFunc,void* pThreadContext, void* pUserData)
	{
		assert(0);
		return (ThreadId)0;
	}
	virtual void WaitForEnd(intptr_t* result)
	{
	}
	virtual void SetName(const char* pName)
	{
	}
	virtual void ChangePriority(int delta)
	{
	}
};

class DefaultThreadLocalStorage : public IThreadLocalStorage
{
public:
	DefaultThreadLocalStorage()
		: m_value(0)
	{

	}
	virtual void* GetValue() 
	{
		return m_value;
	}
	bool SetValue(void* pData)
	{
		m_value = pData;
		return false;
	}
private:
	void* m_value;

};

class DefaultThreadSystem : public IThreadSystem
{
public:
	virtual bool Initialize();
	virtual bool Shutdown();

	virtual IMutex* CreateAMutex();
	virtual void DestroyAMutex(IMutex*);

	virtual IThreadCondition* CreateAThreadCondition();
	virtual void DestroyAThreadCondition(IThreadCondition*);

	virtual IThreadLocalStorage* CreateAThreadLocalStorage();
	virtual void DestroyAThreadLocalStorage(IThreadLocalStorage*);

	virtual IThread* CreateAThread();
	virtual void DestroyAThread(IThread*);

	virtual void ScheduleWork(Telescope::WebKit::ThreadFunc doWork, void* workContext);
	virtual ThreadId CurrentThreadId();
	virtual bool IsMainThread();
	virtual void YieldThread();
	virtual void SleepThread(uint32_t ms);

};

bool DefaultThreadSystem::Initialize()
{
	return true;
}

bool DefaultThreadSystem::Shutdown()
{
	return true;
}

IMutex* DefaultThreadSystem::CreateAMutex()
{
	return new DefaultMutex();
}

void DefaultThreadSystem::DestroyAMutex(IMutex* mutex)
{
	delete mutex;
}

IThreadCondition* DefaultThreadSystem::CreateAThreadCondition()
{
	return new DefaultThreadCondition();
}

void DefaultThreadSystem::DestroyAThreadCondition(IThreadCondition* threadCondition)
{
	delete threadCondition;
}

IThreadLocalStorage* DefaultThreadSystem::CreateAThreadLocalStorage()
{
	return new DefaultThreadLocalStorage();
}

void DefaultThreadSystem::DestroyAThreadLocalStorage(IThreadLocalStorage* threadLocalStorage)
{
	delete threadLocalStorage;
}

IThread* DefaultThreadSystem::CreateAThread()
{
	return new DefaultThread();
}

void DefaultThreadSystem::DestroyAThread(IThread* pThread)
{
	delete pThread;
}

void DefaultThreadSystem::ScheduleWork(Telescope::WebKit::ThreadFunc doWork, void* workContext)
{
	doWork(workContext);
}

Telescope::WebKit::ThreadId DefaultThreadSystem::CurrentThreadId()
{
	return Telescope::WebKit::kThreadIdInvalid;
}

bool DefaultThreadSystem::IsMainThread()
{
	return true;
}

void DefaultThreadSystem::YieldThread()
{
}

void DefaultThreadSystem::SleepThread(uint32_t ms)
{
}


static IThreadSystem* spThreadSystem = NULL; 
void SetThreadSystem(IThreadSystem* pThreadSystem)
{
	assert(!spThreadSystem);
	spThreadSystem = pThreadSystem;
	spThreadSystem->Initialize();
}

IThreadSystem* GetThreadSystem()
{
	if(!spThreadSystem)
	{
		static DefaultThreadSystem defaultThreadSystem;
		spThreadSystem = &defaultThreadSystem;
	}
		
	return spThreadSystem;
}

void ThreadCleanupTick()
{
	WTF::dispatchFunctionsFromMainThread();
	//	may be we should keep this commented line, and implement the function in WebCore
	//WebCore::fireTimerIfNeeded();
	assert(0);
}

}
}
