/*
 * Copyright (C) 2007, 2008, 2015 Apple Inc. All rights reserved.
 * Copyright (C) 2009 Google Inc. All rights reserved.
 * Copyright (C) 2009 Torch Mobile, Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 * 3.  Neither the name of Apple Inc. ("Apple") nor the names of
 *     its contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE AND ITS CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE OR ITS CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * There are numerous academic and practical works on how to implement pthread_cond_wait/pthread_cond_signal/pthread_cond_broadcast
 * functions on Win32. Here is one example: http://www.cs.wustl.edu/~schmidt/win32-cv-1.html which is widely credited as a 'starting point'
 * of modern attempts. There are several more or less proven implementations, one in Boost C++ library (http://www.boost.org) and another
 * in pthreads-win32 (http://sourceware.org/pthreads-win32/).
 *
 * The number of articles and discussions is the evidence of significant difficulties in implementing these primitives correctly.
 * The brief search of revisions, ChangeLog entries, discussions in comp.programming.threads and other places clearly documents
 * numerous pitfalls and performance problems the authors had to overcome to arrive to the suitable implementations.
 * Optimally, WebKit would use one of those supported/tested libraries directly. To roll out our own implementation is impractical,
 * if even for the lack of sufficient testing. However, a faithful reproduction of the code from one of the popular supported
 * libraries seems to be a good compromise.
 *
 * The early Boost implementation (http://www.boxbackup.org/trac/browser/box/nick/win/lib/win32/boost_1_32_0/libs/thread/src/condition.cpp?rev=30)
 * is identical to pthreads-win32 (http://sourceware.org/cgi-bin/cvsweb.cgi/pthreads/pthread_cond_wait.c?rev=1.10&content-type=text/x-cvsweb-markup&cvsroot=pthreads-win32).
 * Current Boost uses yet another (although seemingly equivalent) algorithm which came from their 'thread rewrite' effort.
 *
 * This file includes timedWait/signal/broadcast implementations translated to WebKit coding style from the latest algorithm by
 * Alexander Terekhov and Louis Thomas, as captured here: http://sourceware.org/cgi-bin/cvsweb.cgi/pthreads/pthread_cond_wait.c?rev=1.10&content-type=text/x-cvsweb-markup&cvsroot=pthreads-win32
 * It replaces the implementation of their previous algorithm, also documented in the same source above.
 * The naming and comments are left very close to original to enable easy cross-check.
 *
 * The corresponding Pthreads-win32 License is included below, and CONTRIBUTORS file which it refers to is added to
 * source directory (as CONTRIBUTORS.pthreads-win32).
 */

/*
 *      Pthreads-win32 - POSIX Threads Library for Win32
 *      Copyright(C) 1998 John E. Bossom
 *      Copyright(C) 1999,2005 Pthreads-win32 contributors
 *
 *      Contact Email: rpj@callisto.canberra.edu.au
 *
 *      The current list of contributors is contained
 *      in the file CONTRIBUTORS included with the source
 *      code distribution. The list can also be seen at the
 *      following World Wide Web location:
 *      http://sources.redhat.com/pthreads-win32/contributors.html
 *
 *      This library is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU Lesser General Public
 *      License as published by the Free Software Foundation; either
 *      version 2 of the License, or (at your option) any later version.
 *
 *      This library is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *      Lesser General Public License for more details.
 *
 *      You should have received a copy of the GNU Lesser General Public
 *      License along with this library in the file COPYING.LIB;
 *      if not, write to the Free Software Foundation, Inc.,
 *      59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

 //+TELESCOPE_LIB_CHANGES
// 4/7/2020 3:06 PM
// add place-holder comments for wtf
// 4/8/2020 
// change placeholder function assert of ps & fix some thread assert problems for PS version
// 4/12/2020 
// C++11 thread support of PlayStation for Telescope
// 4/13/2020 
// fix a multi-thread caused exception
// 4/17/2020
// [Telescope][JSC][PS4] use C_LOOP_WIN interpreter for PS4
// 6/18/2020
// [Telescope][WTF] TreatWarningAsError and add 'SCE_LIBC_DISABLE_CPP17_HEADER_WARNING'.

#include "config.h"
#include <wtf/Threading.h>
#include <errno.h>
#include <wtf/HashMap.h>
#include <wtf/Lock.h>
#include <wtf/MainThread.h>
#include <wtf/MathExtras.h>
#include <wtf/NeverDestroyed.h>
#include <wtf/ThreadingPrimitives.h>
#include <thread>
#include "TelescopeWebKitThreadInterface.h"

namespace WTF {

extern double currentTime();
static Lock globalSuspendLock;

//+TELESCOPE_LIB_CHANGES
//8/3/2020
//declare s_threadHolder. remove unused function call in constructor, implement shutdown
extern thread_local Thread::ThreadHolder s_threadHolder;
Thread::ThreadHolder::~ThreadHolder()
{
	if ( thread ) {
		thread->didExit();
	}
	thread = nullptr;
}

void Thread::ThreadHolder::shutdown()
{
	if ( s_threadHolder.thread.get() )
	{
		s_threadHolder.thread.get()->didExit();
	}

	s_threadHolder.thread = nullptr;
}
//-TELESCOPE_LIB_CHANGES

Thread::~Thread()
{
    //assert(0);
}

void Thread::initializeCurrentThreadEvenIfNonWTFCreated()
{
}

#pragma pack(push, 8)
typedef struct tagTHREADNAME_INFO {
    unsigned int dwType;
    char* szName;
    unsigned int dwThreadID;
    unsigned int dwFlags;
} THREADNAME_INFO;
#pragma pack(pop)

void Thread::initializeCurrentThreadInternal(const char* szThreadName)
{
    //assert(0);
}

void Thread::initializePlatformThreading()
{
}

static unsigned wtfThreadEntryPoint(void* data)
{
	Thread::entryPoint(reinterpret_cast<Thread::NewThreadContext*>(data));
    return 0;
}

void Thread::establishHandle(NewThreadContext* data)
{
	std::thread pThread = std::thread(wtfThreadEntryPoint, data);

    m_handle = pThread.native_handle();
	pThread.detach();
}

void Thread::changePriority(int delta)
{
	int priority = delta + SCE_KERNEL_PRIO_FIFO_LOWEST;
	priority = MAX(SCE_KERNEL_PRIO_FIFO_LOWEST, priority);
	priority = MIN(SCE_KERNEL_PRIO_FIFO_HIGHEST, priority);

	scePthreadSetprio(m_handle, priority);
}

int Thread::waitForCompletion()
{
	while (true)
	{
		if (hasExited())
		{
			break;
		}
	}
	
    return 0;
}

void Thread::detach()
{
	scePthreadDetach(m_handle);
}

Expected<void, int> g_dummyRlt2;

auto Thread::suspend() -> Expected<void, PlatformSuspendError>
{
    assert(0);
	LockHolder locker( globalSuspendLock );
    return g_dummyRlt2;
}

void Thread::resume()
{
	assert( 0 );
	LockHolder locker( globalSuspendLock );
}

size_t Thread::getRegisters(PlatformRegisters& registers)
{
    assert(0);
	LockHolder locker( globalSuspendLock );
    return 0;
}

Thread* Thread::currentMayBeNull()
{
	return s_threadHolder.thread.get();
}

Thread& Thread::initializeCurrentTLS()
{
    Ref<Thread> thread = adoptRef(*new Thread());

    thread->initializeInThread();
    initializeCurrentThreadEvenIfNonWTFCreated();

    return initializeTLS(WTFMove(thread));
}

void Thread::establishPlatformSpecificHandle(PlatformThreadHandle hHandle)
{
	//	I don't know why Win's implementation has below code, maybe we don't need it on PlayStation?
	//auto locker = holdLock( m_mutex );
	//m_handle = handle;
	//	m_id = threadID;
}

Thread& Thread::initializeTLS(Ref<Thread>&& thread)
{
	s_threadHolder.thread = WTFMove( thread );
	return *s_threadHolder.thread;
}

Mutex::Mutex()
{
    m_mutex = Telescope::WebKit::GetThreadSystem()->CreateAMutex();
}

Mutex::~Mutex()
{
    Telescope::WebKit::GetThreadSystem()->DestroyAMutex(m_mutex);
}

void Mutex::lock()
{
    m_mutex->Lock();
}

bool Mutex::tryLock()
{
    return m_mutex->TryLock();
}

void Mutex::unlock()
{
    m_mutex->Unlock();
}

ThreadCondition::ThreadCondition()
{
	m_condition = Telescope::WebKit::GetThreadSystem()->CreateAThreadCondition();
}

ThreadCondition::~ThreadCondition()
{
    Telescope::WebKit::GetThreadSystem()->DestroyAThreadCondition( m_condition );
}

void ThreadCondition::wait(Mutex& mutex)
{
    m_condition->Wait(mutex.impl());
}

bool ThreadCondition::timedWait(Mutex& mutex, WallTime absoluteTime)
{
	double currentTime = WTF::currentTime();
	double absoluteTimeRaw = absoluteTime.RawValue();
	if ( absoluteTimeRaw < currentTime )
		return false;

	if ( absoluteTimeRaw - currentTime > static_cast< double >( INT_MAX ) / 1000.0 ) {
		wait( mutex );
		return true;
	}

	double intervalMilliseconds = ( absoluteTimeRaw - currentTime ) * 1000.0;
	return m_condition->TimedWait( mutex.impl(), intervalMilliseconds );

	return true;
}

void ThreadCondition::signal()
{
	m_condition->Signal(false);
}

void ThreadCondition::broadcast()
{
	m_condition->Signal(true);
}


} // namespace WTF

//-TELESCOPE_LIB_CHANGES
