/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "JSONImplementation.h"
#include "TelescopeLib.h"

namespace Telescope { namespace JsonWrapper
{
	//
	// String
	//

	String::String(const RefPtr<WTF::CStringBuffer>& s)
		: m_stringValue(s)
	{}

	void String::Release()
	{
		delete this;
	}

	const char* String::CString() const
	{
		return m_stringValue->data();
	}


	//
	// Value
	//

	Value::Value(const RefPtr<JValue>& value)
		: m_value(value)
	{
	}

	void Value::Release()
	{
		delete this;
	}

	bool Value::GetBoolean(bool& v) const
	{
		return m_value->asBoolean(v);
	}

	bool Value::GetInteger(int& v) const
	{
		return m_value->asInteger(v);
	}

	bool Value::GetInteger(unsigned& v) const
	{
		return m_value->asInteger(v);
	}

	bool Value::GetInteger(long& v) const
	{
		return m_value->asInteger(v);
	}

	bool Value::GetInteger(long long& v) const
	{
		return m_value->asInteger(v);
	}

	bool Value::GetInteger(unsigned long& v) const
	{
		return m_value->asInteger(v);
	}

	bool Value::GetInteger(unsigned long long& v) const
	{
		return m_value->asInteger(v);
	}

	bool Value::GetDouble(double& v) const
	{
		return m_value->asDouble(v);
	}

	bool Value::GetFloat(float& v) const
	{
		return m_value->asDouble(v);
	}

	bool Value::GetString(IString*& v) const
	{
		assert(!v);

		WTF::String s;
		if (!m_value->asString(s))
		{
			v = nullptr;
			return false;
		}
		else
		{
			v = new String(s.ascii().bufferRefPtr());
			return true;
		}
	}

	StringPtr Value::GetString() const
	{
		IString* pString = nullptr;
		GetString(pString);

		return StringPtr(pString);
	}

	ObjectPtr Value::AsObject() const
	{
		RefPtr<JObject> obj;
		if (!m_value->asObject(obj))
			return ObjectPtr();
		else
			return ObjectPtr(new Object(obj));
	}

	ArrayPtr Value::AsArray() const
	{
		RefPtr<JArray> a;
		if (!m_value->asArray(a))
			return ArrayPtr();
		else
			return ArrayPtr(new Array(a));
	}


	//
	// Object
	//

	Object::Object(const RefPtr<JObject>& obj)
		: m_object(obj)
	{
	}

	void Object::Release()
	{
		delete this;
	}

	ValuePtr Object::GetValue(const char* pName) const
	{
		RefPtr<JValue> v;
		if (!m_object->getValue(pName, v))
			return ValuePtr();
		else
			return ValuePtr(new Value(v));
	}

	ObjectPtr Object::GetObject(const char* pName) const
	{
		RefPtr<JObject> obj;
		if (!m_object->getObject(pName, obj))
			return ObjectPtr();
		else
			return ObjectPtr(new Object(obj));
	}

	ArrayPtr Object::GetArray(const char* pName) const
	{
		RefPtr<JArray> a;
		if (!m_object->getArray(pName, a))
			return ArrayPtr();
		else
			return ArrayPtr(new Array(a));
	}

	Object::IteratorPtr Object::GetIterator() const
	{
		return IteratorPtr(new Iterator(m_object));
	}

	// Object::Iterator

	Object::Iterator::Iterator(const RefPtr<JObject>& obj)
		: m_iter(obj->begin())
		, m_iterEnd(obj->end())
	{}

	void Object::Iterator::Release()
	{
		delete this;
	}

	bool Object::Iterator::End() const
	{
		return m_iter == m_iterEnd;
	}

	void Object::Iterator::Next()
	{
		if (m_iter != m_iterEnd)
			++m_iter;
	}

	const char* Object::Iterator::Name() const
	{
		m_name = m_iter->key.ascii().bufferRefPtr();

		return m_name->data();
	}

	ValuePtr Object::Iterator::Value() const
	{
		if (m_iter == m_iterEnd)
			return nullptr;

		return new JsonWrapper::Value(m_iter->value);
	}


	//
	// Array
	//

	Array::Array(const RefPtr<JArray>& a)
		: m_array(a)
	{}

	void Array::Release()
	{
		delete this;
	}

	size_t Array::Length() const
	{
		return m_array->length();
	}

	ValuePtr Array::GetElement(size_t index) const
	{
		if (index >= m_array->length())
			return nullptr;

		return new Value(m_array->get(index));
	}
} // namespace JsonWrapper

	using namespace JsonWrapper;

	//
	// Factory method
	//

	ValuePtr TelescopeLib::ParseJson(const char* pJsonInput, size_t jsonLength)
	{
		if (jsonLength == 0)
			jsonLength = strlen(pJsonInput);

		WTF::String json(pJsonInput, jsonLength);

		RefPtr<JValue> value;
		if (!JValue::parseJSON(json, value))
			return ValuePtr();
		else
			return ValuePtr(new Value(value));
	}

} // namespace Telescope