/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "PlatformFeatureDefs.h"
#include "config.h"
#include "TelescopeJSGlobal.h"
#include "TelescopeJavascriptValue.h"
#include "TelescopePropertyIterator.h"
#include "ClientWrappers/ErrorHandlingReplacement/ErrorHandlingReplacement.h"
#include "Universal/TelescopeLog.h"


namespace Telescope
{
#define TEMP_BUF_LEN	128
	char g_caTempBufLen[TEMP_BUF_LEN];

	TelescopeJSValue::TelescopeJSValue()
		: m_pGlobalObject( nullptr )
		, m_pIterator( nullptr )
	{
		new ( m_caJSValue ) JSC_Value();
	}


	TelescopeJSValue::TelescopeJSValue( const TelescopeJSValue& rSrc )
	{
		m_pIterator = nullptr;
		new ( m_caJSValue ) JSC_Value();
		Copy( rSrc );
	}


	TelescopeJSValue::TelescopeJSValue( JSC_Value* pVal, JSC_GlobalObject* pGlobalObject )
		: m_pGlobalObject( pGlobalObject )
		, m_pIterator( nullptr )
	{
		new ( m_caJSValue ) JSC_Value();
		Assign( pVal );
	}


	TelescopeJSValue::~TelescopeJSValue()
	{
		Destruct();
	}


	void TelescopeJSValue::operator=( const TelescopeJSValue& rSrc )
	{
		Copy( rSrc );
	}


	void TelescopeJSValue::Destruct()
	{
		SAFE_DELETE( m_pIterator );

		JSC::gcUnprotect( *GetImpl() );
		GetImpl()->~JSValue();
	}


	void TelescopeJSValue::Copy( const TelescopeJSValue& rSrc )
	{
		m_pGlobalObject = rSrc.m_pGlobalObject;
		Assign( rSrc.GetImpl() );
	}


	void TelescopeJSValue::Assign( const JSC_Value* pVal )
	{
		JSC::JSLockHolder lock(m_pGlobalObject);

		JSC_Value* pImplValue = GetImpl();
		JSC::gcUnprotect( *pImplValue );
		*pImplValue = *pVal;

		if ( WebCore::commonVMOrNull() && WebCore::commonVMOrNull()->currentThreadIsHoldingAPILock() )
			JSC::gcProtect( *pImplValue );
	}


	void TelescopeJSValue::SetNull()
	{
		JSC_Value oVal = JSC::jsNull();
		Assign( &oVal );
	}

	void TelescopeJSValue::SetUndefined()
	{
		JSC_Value oVal = JSC::jsUndefined();
		Assign( &oVal );
	}

	void TelescopeJSValue::SetNumberValue( double dblVal )
	{
		JSC_Value oVal = JSC::jsNumber( dblVal );
		Assign( &oVal );
	}

	void TelescopeJSValue::SetNumberValue( int iVal )
	{
		JSC_Value oVal = JSC::jsNumber( iVal );
		Assign( &oVal );
	}

	double TelescopeJSValue::GetNumberValue() const
	{
		assert( GetImpl()->isNumber() );
		return GetImpl()->asNumber();
	}

	bool TelescopeJSValue::IsInt32() const
	{
		return GetImpl()->isInt32();
	}

	int TelescopeJSValue::GetInt32() const
	{
		assert( GetImpl()->isInt32() );
		return GetImpl()->asInt32();
	}

	bool TelescopeJSValue::IsDouble() const
	{
		return  GetImpl()->isDouble();
	}

	double TelescopeJSValue::GetDouble() const
	{
		assert( GetImpl()->isDouble() );
		return  GetImpl()->asDouble();
	}

	void TelescopeJSValue::SetBooleanValue( bool bVal )
	{
		JSC_Value oVal = JSC::jsBoolean( bVal );
		Assign( &oVal );
	}

	bool TelescopeJSValue::GetBooleanValue() const
	{
		assert( GetImpl()->isBoolean() );
		return  GetImpl()->asBoolean();
	}

	void TelescopeJSValue::SetStringValue(const void* pVal, bool isWideChar)
	{
		assert(pVal);

		JSC::JSLockHolder lock(m_pGlobalObject);

		WTF::String sVal;
		if (isWideChar)
		{
			sVal = WTF::String(reinterpret_cast<const UChar*>(pVal));
		}
		else
		{
			sVal = WTF::String(reinterpret_cast<const LChar*>(pVal));
		}

		JSC_Value oVal = JSC::jsString(*(WebCore::commonVMOrNull()), sVal);
		Assign(&oVal);
	}

	bool TelescopeJSValue::GetStringValue(void* pInBuf, unsigned int& inoutBufSize, bool& outWideChar) const
	{
		assert( GetImpl()->isString() );
		
		unsigned int inBufLen = inoutBufSize;
		WTF::String&& rrStrArg = GetImpl()->getString(m_pGlobalObject);
		outWideChar = !rrStrArg.is8Bit();
		int terminatorTakesSize = outWideChar ? sizeof(UChar) : sizeof(LChar);
		inoutBufSize = rrStrArg.sizeInBytes() + terminatorTakesSize;

		assert(rrStrArg.sizeInBytes() % terminatorTakesSize == 0);
		if (inoutBufSize > inBufLen)
		{
			ErrorHandling()->RaiseError(LogChannel::GENERAL, Errors::OUT_OF_BUFFER, "GetStringValue input buffer not big enough");

			return false;
		}
		
		memcpy_s(pInBuf,
			inBufLen,
			outWideChar ? static_cast<const void*>(rrStrArg.characters16()) : static_cast<const void*>(rrStrArg.characters8()),
			rrStrArg.sizeInBytes());

		if (outWideChar)
		{
			static_cast<UChar*>(pInBuf)[rrStrArg.sizeInBytes() / terminatorTakesSize] = 0;
		}
		else
		{
			static_cast<LChar*>(pInBuf)[rrStrArg.sizeInBytes() / terminatorTakesSize] = 0;
		}

		return true;
	}

	void TelescopeJSValue::SetArraySize( size_t stSize )
	{
		JSC::JSLockHolder lock(m_pGlobalObject);
		JSC_Value oVal( JSC::constructEmptyArray( m_pGlobalObject, 0, stSize ) );
		Assign( &oVal );
	}

	void TelescopeJSValue::SetArrayValue( size_t stIndex, const TelescopeJSValue& rVal )
	{
		ValToArr( m_pGlobalObject, GetImpl() )->putDirectIndex( m_pGlobalObject, stIndex, *( rVal.GetImpl() ) );
	}

	void TelescopeJSValue::SetArrayValues( const TelescopeJSValue* pVal, size_t stCnt, size_t stStartIndex )
	{
		JSC_Array* pArr = ValToArr( m_pGlobalObject, GetImpl() );
		for ( size_t i = 0; i < stCnt; ++i )
			pArr->putDirectIndex( m_pGlobalObject, stStartIndex + i, *( pVal[i].GetImpl() ) );
	}

	void TelescopeJSValue::PushArrayValue( const TelescopeJSValue& rVal )
	{
		ValToArr( m_pGlobalObject, GetImpl() )->push( m_pGlobalObject, *( rVal.GetImpl() ) );
	}

	void TelescopeJSValue::PopArrayValue( TelescopeJSValue* pValOut )
	{
		if ( !pValOut )
			return;

		JSC_Value oLast = ValToArr( m_pGlobalObject, GetImpl() )->pop( m_pGlobalObject );
		*pValOut = TelescopeJSValue( &oLast, m_pGlobalObject );
	}

	void TelescopeJSValue::GetArrayValue( size_t stIndex, TelescopeJSValue* pValOut ) const
	{
		if ( !pValOut )
			return;

		JSC_Value oVal = ValToArr( m_pGlobalObject, GetImpl() )->getIndex( m_pGlobalObject, stIndex );
		*pValOut = TelescopeJSValue( &oVal, m_pGlobalObject );
	}

	size_t TelescopeJSValue::GetArrayLength() const
	{
		return ValToArr( m_pGlobalObject, GetImpl() )->length();
	}

	void TelescopeJSValue::SetObjectType()
	{
		JSC::JSLockHolder lock(m_pGlobalObject);
		JSC_Value oVal( JSC::constructEmptyObject( m_pGlobalObject ) );
		Assign( &oVal );
	}

	void TelescopeJSValue::SetProperty( const char* pKey, const TelescopeJSValue& rVal )
	{
		JSC_Value oJSValue = *( rVal.GetImpl() );
		JSC::PutPropertySlot oSlot( oJSValue );
		JSC_Object::put(
			ValToObj( GetImpl() ),
			m_pGlobalObject,
			JSC::Identifier::fromString( m_pGlobalObject->vm(), pKey ),
			oJSValue,
			oSlot );
	}

	void TelescopeJSValue::GetProperty( const char* pKey, TelescopeJSValue* pValOut ) const
	{
		JSC_Value oVal = ValToObj( GetImpl() )->get( m_pGlobalObject,
				JSC::Identifier::fromString( m_pGlobalObject->vm(), pKey ) );

		if ( pValOut )
			*pValOut = TelescopeJSValue( &oVal, m_pGlobalObject );
	}

	JSValuePropertyIterator* TelescopeJSValue::First()
	{
		if ( !m_pIterator )
			m_pIterator = new JSValuePropertyIterator( this );

		JSC::PropertyNameArray* pNames = reinterpret_cast< JSC::PropertyNameArray* >( ( void* )m_pIterator->m_caNames );
		JSC_Object::getPropertyNames( ValToObj( GetImpl() ), m_pGlobalObject, *pNames, JSC::EnumerationMode() );
		if ( !pNames->size() )
			return nullptr;

		m_pIterator->m_pItr = ( void* )pNames->begin();
		return m_pIterator;
	}

	JSValuePropertyIterator* TelescopeJSValue::GetNext()
	{
		if ( !m_pIterator )
			return nullptr;

		JSC::PropertyNameArray::const_iterator ciProperty = reinterpret_cast< JSC::PropertyNameArray::const_iterator >( m_pIterator->m_pItr );
		if ( !ciProperty )
			return nullptr;

		++ciProperty;
		JSC::PropertyNameArray* pNames = reinterpret_cast< JSC::PropertyNameArray* >( ( void* )m_pIterator->m_caNames );
		if ( ciProperty == pNames->end() )
		{
			m_pIterator->m_pItr = nullptr;
			return nullptr;
		}

		m_pIterator->m_pItr = ( void* )ciProperty;
		return m_pIterator;
	}


	JSC_Value* TelescopeJSValue::GetImpl() const
	{
		return ( JSC_Value* )&m_caJSValue;
	}


	bool TelescopeJSValue::Call( TelescopeJSValue* pArgs, size_t stCnt, TelescopeJSValue* pValOut )
	{
		if ( pValOut )
			pValOut->SetUndefined();

		JSC_Function* pFunction = ValToFunc( GetImpl() );
		JSC::CallData oCallData;
		JSC::CallType oCallType = JSC_Function::getCallData( pFunction, oCallData );

		if ( oCallType == JSC::CallType::None )
			return false;

		JSC::MarkedArgumentBuffer oArgs;
		if ( pArgs )
		{
			for ( size_t i = 0; i < stCnt; ++i )
				oArgs.append( *( pArgs[i].GetImpl() ) );
		}
		JSC_Value oFunction = *GetImpl();
		JSC_Value oRlt = JSC::call( m_pGlobalObject, oFunction, oCallType, oCallData, oFunction, oArgs );

		if ( m_pGlobalObject->vm().exceptionForInspection() )
			ASSERT_RETURN_FALSE

			if ( pValOut )
			{
				pValOut->SetGlobalObject( m_pGlobalObject );
				pValOut->Assign( &oRlt );
			}

		return true;
	}

	EJSValueType TelescopeJSValue::Type() const
	{
		JSC_Value oVal = *GetImpl();
		if ( oVal.isNumber() )
			return EJVT_Number;
		else if ( oVal.isBoolean() )
			return EJVT_Boolean;
		else if ( oVal.isString() )
			return EJVT_StringType;
		else if ( oVal.isObject() )
		{
			if ( oVal.getObject()->inherits( *( WebCore::commonVMOrNull() ), JSC_Array::info() ) )
				return EJVT_Array;
			else if ( oVal.getObject()->inherits( *( WebCore::commonVMOrNull() ), JSC_Function::info() ) )
				return EJVT_Function;
			else
				return EJVT_Object;
		}
		else if ( oVal.isUndefined() )
			return EJVT_Undefined;
		else if ( oVal.isNull() )
			return EJVT_Null;
		else
			return EJVT_Unknown;
	}

	void TelescopeJSValue::SetGlobalObject( JSC_GlobalObject* pGlobalObject )
	{
		m_pGlobalObject = pGlobalObject;
	}

	JSC_GlobalObject* TelescopeJSValue::GetGlobalObject()
	{
		return m_pGlobalObject;
	}
}
