/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "MemoryTrackingScope.h"

#if TELESCOPE_USING(TELESCOPE_MEMORY_TRACING )

#include <atomic>
#include <utility>
#include <cassert>

#if TELESCOPE_USING(TELESCOPE_PC_PROGRAM) || TELESCOPE_USING(TELESCOPE_XB_PROGRAM)
#include <wtf\Threading.h>
#elif TELESCOPE_USING(TELESCOPE_PS_PROGRAM)
#include <pthread.h>
#endif

namespace Telescope
{
	//
	// MemoryScopeTracker
	//

	// It is a lock-free, thread-safe and lightweight implementation.
	// However:
	//	- The max tracking count of threads is limited by MemoryScopeTracker::MaxTrackingThreads
	//	- The max depth of the scope stack is limited by MemoryScopeTracker::ScopeStack::MaxDepth
	//	- The max length of the scope name is limited by MemoryScopeTracker::ScopeStack::ScopeSize

	class MemoryScopeTracker : public IMemoryScopeTracker
	{
		typedef int ThreadId;

		class ScopeStack
		{
			static constexpr size_t MaxDepth = 16;
			static constexpr size_t ScopeSize = 64;

			ThreadId m_threadId;
			char m_stack[MaxDepth][ScopeSize];
			size_t m_size;
			size_t m_excessCount;

		public:
			ScopeStack()
				: m_threadId(0)
				, m_size(0)
				, m_excessCount(0)
			{}

			void Initialize(ThreadId threadId)
			{
				assert(threadId);
				assert(!m_threadId);

				m_threadId = threadId;
			}

			void Push(const char* pName)
			{
				if (m_size == MaxDepth)
				{
					// As MemoryScopeTracker uses a fixed-size stack,
					// if more scopes are pushed into the stack after MaxDepth has been reached,
					// use m_excessCount to count all subsequent Push calls and match corresponding Pop calls.
					// All these scopes can be lost.

					++m_excessCount;
				}
				else
				{
					auto pTop = m_stack[m_size++];

					CopyString_Truncated(pTop, ScopeSize, pName);
				}
			}

			void Pop()
			{
				assert(m_size || m_excessCount);

				if (m_excessCount)
				{
					--m_excessCount;
				}
				else
				{
					--m_size;
				}
			}

			ThreadId ThreadId() const
			{
				return m_threadId;
			}

			size_t Size() const
			{
				return m_size;
			}

			const char* Scope(size_t index) const
			{
				return index >= m_size ? nullptr : m_stack[index];
			}

		private:
			static void CopyString_Truncated(char* pDest, size_t sizeOfDest, const char* pSrc)
			{
				--sizeOfDest;

				size_t copyLength = std::min(strlen(pSrc), sizeOfDest);

				memcpy_s(pDest, sizeOfDest, pSrc, copyLength);

				pDest[copyLength] = '\0';
			}
		};

		static constexpr size_t MaxTrackingThreads = 8;

		ScopeStack m_stacks[MaxTrackingThreads]; // Threads hava their own instances of ScopeStack
		std::atomic<size_t> m_stackCreateRequestsCount;

	public:
		MemoryScopeTracker()
			: m_stackCreateRequestsCount(0)
		{}

		virtual void* PushScope(const char* pName) override
		{
			auto pStack = GetOrCreateCurrentThreadStack();
			if (pStack)
			{
				pStack->Push(pName);
			}

			// Return the pointer of the stack as a context pointer.
			// So that we can save the lookup time for the stack when PopScope executes
			return pStack;
		}

		virtual void PopScope(void* pStack) override
		{
			if (pStack)
			{
				static_cast<ScopeStack*>(pStack)->Pop();
			}
		}

		virtual void GetScopeStack(const void*& pOutStack, size_t& outCount) const override
		{
			auto pStack = FindStack(GetCurrentThreadId());

			if (pStack)
			{
				pOutStack = pStack;
				outCount = pStack->Size();
			}
			else
			{
				pOutStack = nullptr;
				outCount = 0;
			}
		}

		virtual const char* GetScope(const void* pStack, size_t index) const override
		{
			if (!pStack)
			{
				return nullptr;
			}

			auto p = static_cast<const ScopeStack*>(pStack);

			return index < p->Size() ?
				p->Scope(index) :
				nullptr;
		}

	private:
		ScopeStack* GetOrCreateCurrentThreadStack()
		{
			auto threadId = GetCurrentThreadId();

			auto pStack = FindStack(threadId);
			if (!pStack)
			{
				// Try to add a new stack
				size_t index = m_stackCreateRequestsCount++;
				if (index < MaxTrackingThreads)
				{
					pStack = m_stacks + index;

					pStack->Initialize(threadId);
				}
			}

			return pStack;
		}

		ScopeStack* FindStack(ThreadId threadId) const
		{
			size_t size = std::min(static_cast<size_t>(m_stackCreateRequestsCount), MaxTrackingThreads);

			for (size_t i = 0; i < size; i++)
			{
				if (m_stacks[i].ThreadId() == threadId)
				{
					return const_cast<ScopeStack*>(m_stacks + i);
				}
			}

			return nullptr;
		}

		static ThreadId GetCurrentThreadId()
		{
#if TELESCOPE_USING(TELESCOPE_PC_PROGRAM) || TELESCOPE_USING(TELESCOPE_XB_PROGRAM)
			return Thread::currentID();
#elif TELESCOPE_USING(TELESCOPE_PS_PROGRAM)
			return scePthreadGetthreadid();
#else
#error Implement GetCurrentThreadId() on current platform
#endif
		}
	};
}

#else

namespace Telescope
{
	//
	// MemoryScopeTracker
	//

	class MemoryScopeTracker : public IMemoryScopeTracker
	{
	public:
		virtual void* PushScope(const char*) override { return nullptr; }
		virtual void PopScope(void*) override {}
		virtual void GetScopeStack(const void*&, size_t&) const override {}
		virtual const char* GetScope(const void*, size_t) const override { return nullptr; }
	};
}

#endif


//
// TelescopeLib::GetMemoryScopeTracker
//

namespace Telescope
{
	IMemoryScopeTracker& TelescopeLib::GetMemoryScopeTracker()
	{
		static MemoryScopeTracker singleton;
		return singleton;
	}
}
