/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "PlatformFeatureDefs.h"
#include "Universal/TelescopeAssert.h"
#include "ClientWrappers/MemoryAllocatorBase.h"
#include "TelescopeSubSystemAllocator.h"

#include "wtf/FastMalloc.h"
#include "libxml/xmlmemory.h"
#include "unicode/uclean.h"
#include "jpeglib.h"
#include "freetype/ftsystem.h"

namespace WTF
{
	void* fastMalloc(size_t size, const char* pTag)
	{
		if (size == 0)
		{
			CRASH();
		}
		return Telescope::Telescope_GetMemoryAllocator()->Allocate(size, pTag);
	}


	void* fastZeroedMalloc(size_t size, const char* pTag)
	{
		void* p = fastMalloc(size, pTag);
		memset(p, 0, size);
		return p;
	}


	void* fastAlignedMalloc(size_t alignment, size_t size, const char* pTag)
	{
		if (size == 0)
		{
			CRASH();
		}
		return Telescope::Telescope_GetMemoryAllocator()->AllocateAligned(size, alignment, pTag);
	}


	void* fastCalloc(size_t numElement, size_t elementSize, const char* pTag)
	{
		return fastZeroedMalloc(numElement * elementSize, pTag);
	}


	void* fastRealloc(void* p, size_t size, const char* pTag)
	{
		return Telescope::Telescope_GetMemoryAllocator()->Reallocate(p, size, pTag);
	}


	size_t fastSize(void* p)
	{
		return Telescope::Telescope_GetMemoryAllocator()->AllocateSize(p);
	}


	char* fastStrDup(const char* str, const char* pTag)
	{
		size_t strSize = strlen(str) + 1;
		char* dupStr = static_cast<char*>(fastZeroedMalloc(strSize, pTag));
		memcpy(dupStr, str, strSize);
		return dupStr;
	}


	void fastFree(void* p)
	{
		Telescope::Telescope_GetMemoryAllocator()->Deallocate(p);
	}


	void fastAlignedFree(void* p)
	{
		Telescope::Telescope_GetMemoryAllocator()->DeallocateAligned(p);
	}

	TryMallocReturnValue tryFastMalloc(size_t size, const char* pTag)
	{
		return fastMalloc(size, pTag);
	}


	TryMallocReturnValue tryFastZeroedMalloc(size_t size, const char* pTag)
	{
		return fastZeroedMalloc(size, pTag);
	}


	TryMallocReturnValue tryFastCalloc(size_t numElements, size_t elementSize, const char* pTag)
	{
		return fastCalloc(numElements, elementSize, pTag);
	}


	TryMallocReturnValue tryFastRealloc(void* p, size_t size, const char* pTag)
	{
		return fastRealloc(p, size, pTag);
	}


	void* tryFastAlignedMalloc(size_t alignment, size_t size, const char* pTag)
	{
		return fastAlignedMalloc(alignment, size, pTag);
	}


	size_t fastMallocSize(const void* p)
	{
		return fastSize(const_cast<void*>(p));
	}


	size_t fastMallocGoodSize(size_t bytes)
	{
		TELESCOPE_FAIL_MSG("fastMallocGoodSize not supported\n");
		return 0;
	}


	void fastCommitAlignedMemory(void*, size_t)
	{ // Nothing to do here.
	}


	void fastDecommitAlignedMemory(void*, size_t)
	{ // Nothing to do here.
	}


	void releaseFastMallocFreeMemory()
	{ // Nothing to do here.
	}


	void releaseFastMallocFreeMemoryForThisThread()
	{
		TELESCOPE_FAIL_MSG("releaseFastMallocFreeMemoryForThisThread not supported\n");
	}


	void fastEnableMiniMode()
	{ // Nothing to do here.
	}


	void fastMallocDumpMallocStats()
	{
		TELESCOPE_FAIL_MSG("fastMallocDumpMallocStats not supported\n");
	}


	FastMallocStatistics fastMallocStatistics()
	{
		TELESCOPE_FAIL_MSG("fastMallocStatistics not supported\n");
		return FastMallocStatistics();
	}


	void fastSetMaxSingleAllocationSize(size_t size)
	{ // Nothing to do
	}


	// Problematic
	bool isFastMallocEnabled()
	{
		TELESCOPE_FAIL_MSG("isFastMallocEnabled not supported\n");
		return false;
	}
} // End of namespace WTF


extern "C"
{
	// cairo
	#define CAIRO_MEMORY_TAG "cairo"

	void* _cairo_malloc_customized(size_t size)
	{
		return WTF::fastMalloc(size, CAIRO_MEMORY_TAG);
	}


	void* _cairo_calloc_customized(size_t num, size_t size)
	{
		return WTF::fastCalloc(num, size, CAIRO_MEMORY_TAG);
	}


	void* _cairo_realloc_customized(void* p, size_t size)
	{
		return WTF::fastRealloc(p, size, CAIRO_MEMORY_TAG);
	}


	char* _cairo_strdup_customized(const char* s)
	{
		return WTF::fastStrDup(s, CAIRO_MEMORY_TAG);
	}


	void _cairo_free_customized(void* p)
	{
		WTF::fastFree(p);
	}


	// jpeg
	#define JPEG_MEMORY_TAG "jpeg"

	void* jpeg_get_small(j_common_ptr cinfo, size_t sizeofobject)
	{
		return WTF::fastMalloc(sizeofobject, JPEG_MEMORY_TAG);
	}


	void jpeg_free_small(j_common_ptr cinfo, void* object, size_t sizeofobject)
	{
		WTF::fastFree(object);
	}


	void* jpeg_get_large(j_common_ptr cinfo, size_t sizeofobject)
	{
		return WTF::fastMalloc(sizeofobject, JPEG_MEMORY_TAG);
	}


	void jpeg_free_large(j_common_ptr cinfo, void* object, size_t sizeofobject)
	{
		WTF::fastFree(object);
	}


	// pixman
	#define PIXMAN_MEMORY_TAG  "pixman"

	void* pixman_malloc(size_t size)
	{
		return WTF::fastMalloc(size, PIXMAN_MEMORY_TAG);
	}


	void* pixman_calloc(size_t num, size_t size)
	{
		return WTF::fastCalloc(num, size, PIXMAN_MEMORY_TAG);
	}


	void pixman_free(void* p)
	{
		WTF::fastFree(p);
	}


	// FreeType
	#define FREETYPE_MEMORY_TAG "freetype"

	void* ft_custom_alloc(FT_Memory memory, long size)
	{
		return WTF::fastMalloc(size, FREETYPE_MEMORY_TAG);
	}


	void* ft_custom_realloc(FT_Memory memory, long cur_size, long new_size, void* block)
	{
		return WTF::fastRealloc(block, new_size, FREETYPE_MEMORY_TAG);
	}


	void ft_custom_free(FT_Memory memory, void* block)
	{
		WTF::fastFree(block);
	}


	// Font Config
	#define FONT_CONFIG_MEMORY_TAG "fontconfig"

	void* Fc_customized_malloc(size_t size)
	{
		return WTF::fastMalloc(size, FONT_CONFIG_MEMORY_TAG);
	}


	void* Fc_customized_realloc(void* p, size_t size)
	{
		return WTF::fastRealloc(p, size, FONT_CONFIG_MEMORY_TAG);
	}


	void* Fc_customized_calloc(size_t num, size_t size)
	{
		return WTF::fastCalloc(num, size, FONT_CONFIG_MEMORY_TAG);
	}


	void* Fc_customized_strdup(const char* s)
	{
		return WTF::fastStrDup(s, FONT_CONFIG_MEMORY_TAG);
	}


	void Fc_customized_free(void* p)
	{
		WTF::fastFree(p);
	}


	// harfbuzz
	#define HARFBUZZ_MEMORY_TAG "harfbuzz"

	void* hb_malloc_impl(size_t size)
	{
		return WTF::fastMalloc(size, HARFBUZZ_MEMORY_TAG);
	}


	void* hb_calloc_impl(size_t nmemb, size_t size)
	{
		return WTF::fastCalloc(nmemb, size, HARFBUZZ_MEMORY_TAG);
	}


	void* hb_realloc_impl(void* ptr, size_t size)
	{
		return WTF::fastRealloc(ptr, size, HARFBUZZ_MEMORY_TAG);
	}


	void  hb_free_impl(void* ptr)
	{
		WTF::fastFree(ptr);
	}
};


// xmllib2
#define XMLLIB2_MEMORY_TAG "xmllib2"

void xmlFreeCustomized(void* p)
{
	WTF::fastFree(p);
}


void* xmlMallocCustomized(size_t size)
{
	return WTF::fastMalloc(size, XMLLIB2_MEMORY_TAG);
}


void* xmlReallocCustomized(void* p, size_t size)
{
	return WTF::fastRealloc(p, size, XMLLIB2_MEMORY_TAG);
}


char* xmlStrdupCustomized(const char* str)
{
	return WTF::fastStrDup(str, XMLLIB2_MEMORY_TAG);
}


// icu
#define ICU_MEMORY_TAG "icu"

void* icuMallocCustomized(const void* context, size_t size)
{
	return WTF::fastMalloc(size, ICU_MEMORY_TAG);
}


void* icuReallocCustomized(const void* context, void* mem, size_t size)
{
	return WTF::fastRealloc(mem, size, ICU_MEMORY_TAG);
}


void icuFreeCustomized(const void* context, void* mem)
{
	WTF::fastFree(mem);
}


// sqlite
#define SQLITE_MEMORY_TAG "sqlite"

void* sqLite_malloc(int size)
{
	return WTF::fastMalloc(size, SQLITE_MEMORY_TAG);
}


void sqLite_free(void* p)
{
	WTF::fastFree(p);
}


void* sqLite_realloc(void* p, int size)
{
	return WTF::fastRealloc(p, size, SQLITE_MEMORY_TAG);
}


int sqLite_size(void* p)
{
	return WTF::fastSize(p);
}


int sqLite_roundUp(int size)
{
	return (size + 7) & ~7;
}


int sqLite_init(void*)
{
	return 0;
}


void sqLite_shutDown(void*)
{// Nothing to do
}


namespace Telescope
{
	void Telescope_InstallSubSystemAllocator()
	{
		// libxml2
		xmlMemSetup(xmlFreeCustomized, xmlMallocCustomized, xmlReallocCustomized, xmlStrdupCustomized);
		// icu
		UErrorCode icuErrorCode = U_ZERO_ERROR;
		u_setMemoryFunctions(nullptr, icuMallocCustomized, icuReallocCustomized, icuFreeCustomized, &icuErrorCode);
		if (icuErrorCode != U_ZERO_ERROR)
		{
			TELESCOPE_FAIL_MSG("Fail to install customized memory allocator for ICU, wrong argument. ");
		}
	}
}