/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "PlatformFeatureDefs.h"
#include "Universal/TelescopeLog.h"
#include "TelescopeLib.h"
#include "WebView.h"
#include "ClientWrappers/MemoryAllocatorBase.h"
#include "ClientWrappers/CURLReplacement/CurlReplacementImpl.h"
#include "ClientWrappers/HashingReplacement/HashingReplacement.h"
#include "ClientWrappers/ICUMMAPReplacement/IcuMmpReplacement.h"
#include "ClientWrappers/SystemAPIReplacement/SystemAPIReplacement.h"
#include "ClientWrappers/ErrorHandlingReplacement/ErrorHandlingReplacement.h"
#include "ClientWrappers/LogReplacement/LogReplacement.h"
#include "wtf/MainThread.h"
#include "wtf/RunLoop.h"
#include "wtf/TelescopePrintStream.h"
#include "fontconfig/fontconfig.h"
#include "MainThreadSharedTimer.h"
#include "Logging.h"
#include <wtf/ThreadSpecific.h>
#include "network/curl/CurlShutdown.h"
#include "network/NetworkStateNotifier.h"
#include <WebCore/CurlCacheManager.h>
#include "MemoryTrackingScope.h"

extern void InstallCompressionWrapper(CompressionWrapperBase* pCompressionWrapper );
extern CompressionWrapperBase* getCompressionWrapper();
extern unsigned int g_uiTelescopeCL;

#if IS_DEBUG
#define sPublishVersion "Debug"
#elif IS_RELEASE
#define sPublishVersion "Release"
#elif IS_SHIP
#define sPublishVersion "Ship"
#else
#define sPublishVersion "Unknown"
#endif

namespace Telescope
{

void TelescopeLib::PreInit(TelescopeConfig* config)
{
	// Memory must be the first one to install
	InstallMemoryAllocator(config->m_pMemAllocator);

	InstallErrorHandlingWrapper(config->m_pErrorHandlingImpl);

	ASSERT_WITH_MESSAGE(config->m_pHttpWrapper, "Telescope must have the implementation of HTTP.");
	if (config->m_pHttpWrapper)
	{
		InstallHttpWrapper(config->m_pHttpWrapper);
	}

	ASSERT_WITH_MESSAGE(config->m_pHashingImpl, "Telescope must have the implementation of HASHING.");
	if (config->m_pHashingImpl)
	{
		InstallHashingWrapper(config->m_pHashingImpl);
	}

	ASSERT_WITH_MESSAGE(config->m_pCompressionImpl, "Telescope must have the implementation of a Compression implementation.");
	if (config->m_pCompressionImpl)
	{
		InstallCompressionWrapper(config->m_pCompressionImpl);
	}

	ASSERT_WITH_MESSAGE(config->m_pAssetLoaderImpl, "Telescope must have the implementation to asset loader.");
	if (config->m_pAssetLoaderImpl)
	{
		InstallOpenFileWrapper(config->m_pAssetLoaderImpl);
	}

	ASSERT_WITH_MESSAGE(config->m_pSysAPIImpl, "Telescope must have the implementation of system API.");
	if (config->m_pSysAPIImpl)
	{
		InstallSystemAPIWrapper(config->m_pSysAPIImpl);
	}

	ASSERT_WITH_MESSAGE(config->m_pLogImpl, "Telescope must have the implementation of log system.");
	if (config->m_pLogImpl)
	{
		InstallLogWrapper(config->m_pLogImpl);
	}
}

bool TelescopeLib::Init(TelescopeConfig* config )
{
	WTF::RunLoop::initializeMainRunLoop();

	RELEASE_LOG_INFO( Telescope, "PublishVersion:[%s], Compile Time:[%s %s], CL: %u\n",
		sPublishVersion, __DATE__, __TIME__, g_uiTelescopeCL );

	return true;
}

void TelescopeLib::Shutdown()
{
	// TODO: Looks like Telescope is not shutdown when reaches here.
	// Telescope_UninstallMemoryAllocator();

	WebCore::CurlCacheManager::singleton().saveIndex();

#if TELESCOPE_USING(TELESCOPE_MICROSOFT_PROGRAM)
	NetworkStateNotifier::singleton().Cleanup();
#endif

	WTF::RunLoop::shutdown();

#if USE(PTHREADS)
	WTF::shutdownPThreadKeyHashMap();
#endif

	WebCore::curlShutdown();
}

void TelescopeLib::Tick()
{
	WTF::dispatchFunctionsFromMainThread();

	WebCore::tickTimer();
}

WebView* TelescopeLib::CreateView()
{
	TELESCOPE_MEMORY_SCOPE("TelescopeLib::CreateView");

	return new WebView();
}

void TelescopeLib::DestroyView(WebView* view)
{
	view->ShutdownView();
	delete view;
}

bool TelescopeLib::InstallFont(const unsigned char* fontData, size_t dataSize)
{
	return FcAddFontMemory(fontData, dataSize);
}

void TelescopeLib::InitFileCache(const char* cacheDirectory, size_t storageSizeLimitInBytes)
{
	String cacheFolder(cacheDirectory);
	WebCore::CurlCacheManager::singleton().setCacheDirectory(cacheFolder);
	WebCore::CurlCacheManager::singleton().setStorageSizeLimit(storageSizeLimitInBytes);
}

unsigned int TelescopeLib::getTelescopeCL()
{
	return g_uiTelescopeCL;
}

}
