/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
//#include "WebKitPrefix.h"
#include "WebCore/IntPoint.h"
#include "WebCore/PlatformMouseEvent.h"
#include "WebCore/EventHandler.h"
#include "WebCore/Frame.h"
#include "WebCore/PlatformEvent.h"
#include "WebCore/PlatformWheelEvent.h"
#include "WebCore/Scrollbar.h"
#include "WebCore/ScriptController.h"
#include "WebCore/Page.h"
#include "WebCore/PlatformWheelEvent.h"
#include "WebCore/FocusController.h"
#include "WebCore/PlatformKeyboardEvent.h"
#include "WebCore/Editor.h"
#include "TelescopeOperation.h"
#include "JavaScript/TelescopeJSGlobal.h"
#include "WebCore/PlatformEvent.h"
#include "MemoryTrackingScope.h"

#ifdef _WIN64
#include <windowsx.h>
#endif


namespace Telescope
{
	using namespace WebCore;

	//	TODO: dynamic alloc and release
	static IOperation s_operation;

	IOperation* Telescope_GetOperation()
	{
		return& s_operation;
	}


	void ProcessKeyModifier( uint32_t uModifiers, OptionSet<PlatformEvent::Modifier>& rOut )
	{
		if ( uModifiers & g_uiKModifierMaskShift )
			rOut.add( PlatformEvent::Modifier::ShiftKey );
		if ( uModifiers & g_uiKModifierMaskControl )
			rOut.add( PlatformEvent::Modifier::ControlKey );
		if ( uModifiers & g_uiKModifierMaskAlt )
			rOut.add( PlatformEvent::Modifier::AltKey );
		if ( uModifiers & g_uiKModifierMaskOS )
			rOut.add( PlatformEvent::Modifier::MetaKey );
	}

	static OptionSet<WebCore::PlatformEvent::Modifier> BuildKeyModifier()
	{
		OptionSet<WebCore::PlatformEvent::Modifier> keyModifier;
#if TELESCOPE_USING(TELESCOPE_PC_PROGRAM)
		constexpr unsigned short WIN_HIGH_BIT_MASK_SHORT = 0x8000;
		if (GetKeyState(VK_SHIFT) & WIN_HIGH_BIT_MASK_SHORT)
		{
			keyModifier.add(PlatformEvent::Modifier::ShiftKey);
		}
		if (GetKeyState(VK_CONTROL) & WIN_HIGH_BIT_MASK_SHORT)
		{
			keyModifier.add(PlatformEvent::Modifier::ControlKey);
		}
		if (GetKeyState(VK_MENU) & WIN_HIGH_BIT_MASK_SHORT)
		{
			keyModifier.add(PlatformEvent::Modifier::AltKey);
		}
#elif TELESCOPE_USING(TELESCOPE_XB_PROGRAM)
		//	TODO: Get shift, control, alt status for XBOX
#else
		//	TODO: Get shift, control, alt status for PlayStation
#endif
		return keyModifier;
	}

	static WTF::WallTime BuildWallTime()
	{
#if TELESCOPE_USING(TELESCOPE_PC_PROGRAM) || TELESCOPE_USING(TELESCOPE_XB_PROGRAM)
		return WallTime::fromRawSeconds(::GetTickCount64() * 0.001);
#else
		return WallTime::fromRawSeconds(sceKernelGetProcessTime() * 0.000001);
#endif
	}

	void IOperation::BuildPlatformMouseMoveEvent( const MouseMoveEvent& rEv, WebCore::PlatformMouseEvent* pOut )
	{
		if ( !pOut )
			return;

		pOut->m_position = IntPoint( rEv.m_iX, rEv.m_iY );
		pOut->m_globalPosition = IntPoint( rEv.m_iGlobalX, rEv.m_iGlobalY );

		switch ( rEv.m_uID )
		{
		case g_uiKMouseLeft:
			pOut->m_button = LeftButton;
			break;
		case g_uiKMouseMiddle:
			pOut->m_button = MiddleButton;
			break;
		case g_uiKMouseRight:
			pOut->m_button = RightButton;
			break;
		default:
			pOut->m_button = NoButton;
			break;
		}
		pOut->m_type = PlatformEvent::MouseMoved;
		pOut->m_clickCount = 0;

		ProcessKeyModifier( rEv.m_uModifiers, pOut->m_modifiers );
		pOut->m_modifierFlags = rEv.m_uModifiers;

		pOut->m_timestamp = WallTime::now();
	}

	void IOperation::BuildPlatformMouseButtonEvent( const MouseButtonEvent& rEv, WebCore::PlatformMouseEvent* pOut )
	{
		if ( !pOut )
			return;

		pOut->m_position = IntPoint( rEv.m_iX, rEv.m_iY );
		pOut->m_globalPosition = IntPoint( rEv.m_iGlobalX, rEv.m_iGlobalY );

		switch ( rEv.m_uID )
		{
		case g_uiKMouseLeft:
			pOut->m_button = LeftButton;
			break;
		case g_uiKMouseMiddle:
			pOut->m_button = MiddleButton;
			break;
		case g_uiKMouseRight:
			pOut->m_button = RightButton;
			break;
		default:
			ASSERT_NOT_REACHED();
			pOut->m_button = NoButton;
			break;
		}

		pOut->m_type = rEv.m_bDepressed ? PlatformEvent::MousePressed : PlatformEvent::MouseReleased;

		pOut->m_clickCount = rEv.m_uClickCount;

		ProcessKeyModifier( rEv.m_uModifiers, pOut->m_modifiers );
		pOut->m_modifierFlags = rEv.m_uModifiers;

		pOut->m_timestamp = WallTime::now();
	}

	void IOperation::BuildPlatformMouseWheelEvent( const MouseWheelEvent& rEv, WebCore::PlatformWheelEvent* pOut )
	{
		if ( !pOut )
			return;

		pOut->m_position = IntPoint( rEv.m_iX, rEv.m_iY );
		pOut->m_globalPosition = IntPoint( rEv.m_iGlobalX, rEv.m_iGlobalY );

		ProcessKeyModifier( rEv.m_uModifiers, pOut->m_modifiers );

		pOut->m_wheelTicksX = ( !pOut->shiftKey() ) ? 0 : rEv.m_iZDelta;
		pOut->m_wheelTicksY = ( !pOut->shiftKey() ) ? rEv.m_iZDelta : 0;

		pOut->m_deltaX = pOut->m_wheelTicksX;
		pOut->m_deltaY = pOut->m_wheelTicksY;

		pOut->m_deltaX *= ( rEv.m_iNumLines * WebCore::Scrollbar::pixelsPerLineStep() );
		pOut->m_deltaX *= ( rEv.m_iNumLines * WebCore::Scrollbar::pixelsPerLineStep() );

		pOut->m_granularity = ScrollByPixelWheelEvent;
	}

	JSC_Value evaluateJavaScript( const char* scriptSource, WebCore::Frame* pFrame )
	{
#if TELESCOPE_USING(TELESCOPE_MEMORY_TRACING)
		char name[128] = "evaluateJavaScript ";
		strcat(name, scriptSource);
		TELESCOPE_MEMORY_SCOPE(name);
#endif

		if ( pFrame )
		{
			return pFrame->script().executeScriptIgnoringException( scriptSource );
		}

		return JSC_Value( JSC_Value::JSUndefined );
	}

	int32_t IOperation::evaluateJavaScriptAsInt32( const char* scriptSource, WebCore::Frame* pFrame )
	{
		if ( pFrame )
		{
			JSC_Value jsRlt = evaluateJavaScript( scriptSource, pFrame );
			if ( jsRlt.isUndefinedOrNull() )
				return INT_MAX;

			if ( jsRlt.isInt32() )
				return jsRlt.asInt32();
			else if ( jsRlt.isInt32AsAnyInt() )
				return jsRlt.asInt32AsAnyInt();
		}

		return INT_MAX;
	}

	void IOperation::evaluateJavaScriptAsVoid( const char* scriptSource, WebCore::Frame* pFrame )
	{
		if ( pFrame )
		{
			evaluateJavaScript( scriptSource, pFrame );
		}
	}

	uint32_t IOperation::evaluateJavaScriptAsUInt32( const char* scriptSource, WebCore::Frame* pFrame )
	{
		if ( pFrame )
		{
			JSC_Value jsRlt = evaluateJavaScript( scriptSource, pFrame );
			if ( jsRlt.isUndefinedOrNull() )
				return UINT_MAX;

			if ( jsRlt.isUInt32() )
				return jsRlt.asUInt32();
			else if ( jsRlt.isUInt32AsAnyInt() )
				return jsRlt.asUInt32AsAnyInt();
		}

		return UINT_MAX;
	}

	long long IOperation::evaluateJavaScriptAsAnyInt( const char* scriptSource, WebCore::Frame* pFrame )
	{
		if ( pFrame )
		{
			JSC_Value jsRlt = evaluateJavaScript( scriptSource, pFrame );
			if ( jsRlt.isUndefinedOrNull() )
				return INT64_MAX;

			if ( jsRlt.isAnyInt() )
				return jsRlt.asAnyInt();
		}

		return INT64_MAX;
	}

	double IOperation::evaluateJavaScriptAsDouble( const char* scriptSource, WebCore::Frame* pFrame )
	{
		if ( pFrame )
		{
			JSC_Value jsRlt = evaluateJavaScript( scriptSource, pFrame );
			if ( jsRlt.isUndefinedOrNull() )
				return DBL_MAX;

			if ( jsRlt.isDouble() )
				return jsRlt.asDouble();
			else if ( jsRlt.isNumber() )
				return jsRlt.asNumber();
		}

		return DBL_MAX;
	}

	bool IOperation::evaluateJavaScriptAsBoolean( const char* scriptSource, WebCore::Frame* pFrame, bool* pFailed )
	{
		if ( pFrame )
		{
			JSC_Value jsRlt = evaluateJavaScript( scriptSource, pFrame );
			if ( !jsRlt.isUndefinedOrNull() && jsRlt.isBoolean() )
				return jsRlt.asBoolean();
		}

		if ( pFailed )
			*pFailed = true;

		return false;
	}

	void IOperation::OnMouseMoveEvent( const MouseMoveEvent& rEv, WebCore::Frame* pFrame )
	{
		if ( !pFrame || !pFrame->view() )
			return;

		WebCore::PlatformMouseEvent oEvent;
		BuildPlatformMouseMoveEvent( rEv, &oEvent );

		pFrame->eventHandler().mouseMoved( oEvent );
	}

	void IOperation::OnMouseButtonEvent( const MouseButtonEvent& rEv, WebCore::Frame* pFrame )
	{
		if ( !pFrame || !pFrame->view() )
			return;

		WebCore::PlatformMouseEvent oEvent;
		BuildPlatformMouseButtonEvent( rEv, &oEvent );

		if ( rEv.m_bDepressed )
		{
			pFrame->eventHandler().handleMouseReleaseEvent( oEvent );
		}
		else
		{
			pFrame->eventHandler().handleMousePressEvent( oEvent );
		}
	}

	void IOperation::OnMouseWheelEvent( const MouseWheelEvent& rEv, WebCore::Frame* pFrame )
	{
		if ( !pFrame || !pFrame->view() )
			return;

		WebCore::PlatformWheelEvent oEvent;
		BuildPlatformMouseWheelEvent( rEv, &oEvent );

		pFrame->eventHandler().handleWheelEvent( oEvent );
	}

	void IOperation::ProcessKeyboardKey(unsigned long long wParam, long long lParam, bool bKeyDown, WebCore::Frame* pFrame)
	{
		auto& frame = pFrame->page()->focusController().focusedOrMainFrame();
#if TELESCOPE_USING(TELESCOPE_PC_PROGRAM)
		Vector<MSG> pendingCharEvents;
		MSG msg;
		while (PeekMessage(&msg, nullptr, WM_CHAR, WM_DEADCHAR, PM_REMOVE))
		{
			// FIXME: remove WM_UNICHAR, too
			// WM_SYSCHAR events should not be removed, because WebKit is using WM_SYSCHAR for access keys and they can't be canceled.
			if (msg.message == WM_CHAR)
				pendingCharEvents.append(msg);
		}

		PlatformKeyboardEvent keyEvent(nullptr, wParam, lParam, bKeyDown ? PlatformEvent::RawKeyDown : WebCore::PlatformEvent::KeyUp, false);
		if (frame.eventHandler().keyEvent(keyEvent))
		{
			return;
		}

		for (auto& msg : pendingCharEvents)
		{
			DispatchMessage(&msg);
		}
#else
		PlatformKeyboardEvent keyEvent(
			bKeyDown ? WebCore::PlatformEvent::RawKeyDown : WebCore::PlatformEvent::KeyUp,
			String(),
			String(),
			String(),
			String(),
			String(),
			wParam,
			false,
			false,
			false,
			BuildKeyModifier(),
			BuildWallTime()
		);

		frame.eventHandler().keyEvent(keyEvent);
#endif	//	#if TELESCOPE_USING(TELESCOPE_PC_PROGRAM)
	}

	void IOperation::ProcessKeyboardInputChar(unsigned long long wParam, long long lParam, bool eraseAll, WebCore::Frame* pFrame)
	{
		UChar keyCode = static_cast<UChar>(wParam);
		PlatformKeyboardEvent keyEvent(
			WebCore::PlatformEvent::Char,
			String(&keyCode, 1),
			String(&keyCode, 1),
			String(&keyCode, 1),
			String(&keyCode, 1),
			String(""),
			0,
			false,
			false,
			false,
			BuildKeyModifier(),
			BuildWallTime()
		);

		if (eraseAll)
		{
			pFrame->editor().command("SelectAll").execute();
		}

		pFrame->eventHandler().keyEvent(keyEvent);
	}

	void IOperation::ProcessKeyboardInputString( const char* str, bool eraseAll, WebCore::Frame* pFrame )
	{
		if ( eraseAll )
			pFrame->editor().command( "SelectAll" ).execute();

		PlatformKeyboardEvent keyEvent( PlatformEvent::Char, String( str ), String( str ), "", "", "", 0, false, false, false, OptionSet<PlatformEvent::Modifier>(), WallTime::now() );
		pFrame->eventHandler().keyEvent( keyEvent );
	}

	void IOperation::SetFocus( bool focused, WebCore::Frame* pFrame )
	{
		pFrame->page()->focusController().setFocused( focused );
	}

	void IOperation::OnGainFocus( WebCore::Frame* pFrame )
	{
		evaluateJavaScript( "Telescope_CB_GainFocus();", pFrame );
	}

	void IOperation::OnLoseFocus( WebCore::Frame* pFrame )
	{
		evaluateJavaScript( "Telescope_CB_LoseFocus();", pFrame );
	}

	bool IOperation::OnFocusNextElement( WebCore::Frame* pFrame, int iIndex )
	{
		char caJSCmd[MAX_JS_CMD_LEN] = "";
		sprintf( caJSCmd, "WK_FocusNextElement(%d);", iIndex );

		return evaluateJavaScriptAsBoolean( caJSCmd, pFrame, nullptr );
	}

	void IOperation::OnGamePadPressed( WebCore::Frame* pFrame, int iButtonIndex )
	{
		char caJSCmd[MAX_JS_CMD_LEN] = "";
		sprintf( caJSCmd, "Telescope_CB_GamePadPressed(%d);", iButtonIndex );
		evaluateJavaScript( caJSCmd, pFrame );
	}


	void IOperation::BindJSObject(WebCore::Frame* pFrame, const char* pObjStringVal)
	{
	}
}

