/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "WebView.h"
#include "WebFrame.h"
#include "WebPlatformStrategies.h"
#include "WebViewGroup.h"
#include "WebFrameLoaderClient.h"
#include "WebPluginInfoProvider.h"
#include "WebInspectorClient.h"
#include "WebDatabaseProvider.h"
#include "PageStorageSessionProvider.h"
#include "WebEditorClient.h"
#include "WebProgressTrackerClient.h"
#include "BackForwardList.h"
#include "WebFrameNetworkingContext.h"
#include "WebChromeClient.h"
#include "WebVisitedLinkStore.h"
#include "TelescopeSurface.h"

#include <cairo.h>
#include <webcore/Page.h>
#include <webcore/Frame.h>
#include <webcore/NetworkStorageSession.h>
#include <JavaScriptCore/InitializeThreading.h>
#include <WTF/MemoryPressureHandler.h>
#include <WTF/StdUnorderedMap.h>
#include <WebCore/MemoryRelease.h>
#include <webcore/DeprecatedGlobalSettings.h>
#include <WebCore/PageConfiguration.h>
#include <PAL/SessionID.h>
#include <webcore/SocketProvider.h>
#include <webcore/CacheStorageProvider.h>
#include <webcore/CookieJar.h>
#include <webcore/MediaRecorderPrivate.h>
#include <webcore/LibWebRTCProvider.h>
#include <webcore/MediaRecorderProvider.h>
#include <webcore/MIMETypeRegistry.h>
#include <webcore/ApplicationCacheStorage.h>
#include <webcore/RuntimeEnabledFeatures.h>
#include <webcore/URLUtils.h>
#include <webcore/Settings.h>
#include <webcore/DatabaseManager.h>
#include <WebCore/UserContentController.h>
#include <webcore/StorageNamespaceProvider.h>
#include <webcore/PlatformContextCairo.h>
#include <webcore/ProgressTracker.h>
#include <WebCore/FocusController.h>
#include <WebCore/EventHandler.h>
#include <WebCore/PlatformKeyboardEvent.h>
#include <webcore/EventNames.h>
#include <WebCore/Editor.h>
#include <WebCore/KeyboardEvent.h>
#include <WebCore/GraphicsLayer.h>
#include <fontconfig/fontconfig.h>
#include "PlatformFeatureDefs.h"
#include "Universal/TelescopeLog.h"
#include "TelescopeJavascriptValue.h"
#include "TelescopeAPI.h"
#include "Universal/ViewPrivate.h"
#include "ViewIME.h"
#include "ClientWrappers/LogReplacement/LogReplacement.h"
#include "MemoryTrackingScope.h"

#include "AcceleratedCompositingContext.h"

using namespace WebCore;

namespace Telescope
{
	WebCore::ApplicationCacheStorage& StorageInstance()
	{
		static ApplicationCacheStorage& storage = ApplicationCacheStorage::create( "", "ApplicationCache" ).leakRef();

		return storage;
	}

	WebView::WebView()
	{
		d = new ViewPrivate();

#if TELESCOPE_USING(TELESCOPE_PC_PROGRAM)
		ime = new ViewIME();
#endif

		JSC::initializeThreading();
		WebCore::NetworkStorageSession::permitProcessToUseCookieAPI( true );

		d->mWebViewGroup->addWebView( this );
	}

	WebView::~WebView()
	{
		if ( d->mPage )
		{
			delete d->mPage;
		}
		delete d;

#if TELESCOPE_USING(TELESCOPE_PC_PROGRAM)
		delete ime;
#endif
	}

	void WebView::ApplyTelescopeDefaultSettings()
	{
		auto settings = &d->mPage->settings();

		settings->setStandardFontFamily( "Times New Roman" );
		settings->setFixedFontFamily( "Courier New" );
		settings->setSerifFontFamily( "Times New Roman" );
		settings->setSansSerifFontFamily( "Arial" );
		settings->setCursiveFontFamily( "Comic Sans MS" );
		settings->setFantasyFontFamily( "Comic Sans MS" );
		settings->setPictographFontFamily( "Segoe UI Symbol" );
		settings->setDefaultFontSize( 16 );
		settings->setDefaultFixedFontSize( 13 );
		settings->setMinimumLogicalFontSize( 9 );
		settings->setMinimumFontSize( 0 );
		settings->setDefaultTextEncodingName( "ISO-8859-1" );
		settings->setFontRenderingMode( FontRenderingMode::Normal );
#if ENABLE(VIDEO_TRACK)
		settings->setShouldDisplaySubtitles( false );
		settings->setShouldDisplayCaptions( false );
		settings->setShouldDisplayTextDescriptions( false );
#endif

		settings->setJavaEnabled( true );
		settings->setScriptEnabled( true );
		settings->setPluginsEnabled( true );

#if ENABLE(FETCH_API)
		RuntimeEnabledFeatures::sharedFeatures().setFetchAPIEnabled( false );
#endif

		RuntimeEnabledFeatures::sharedFeatures().setShadowDOMEnabled( false );
		RuntimeEnabledFeatures::sharedFeatures().setCustomElementsEnabled( false );
		RuntimeEnabledFeatures::sharedFeatures().setMenuItemElementEnabled( false );
		RuntimeEnabledFeatures::sharedFeatures().setKeygenElementEnabled( false );
		RuntimeEnabledFeatures::sharedFeatures().setModernMediaControlsEnabled( false );
		RuntimeEnabledFeatures::sharedFeatures().setWebAnimationsEnabled( true );
		RuntimeEnabledFeatures::sharedFeatures().setWebAnimationsCompositeOperationsEnabled( false );
		RuntimeEnabledFeatures::sharedFeatures().setWebAnimationsMutableTimelinesEnabled( false );
		RuntimeEnabledFeatures::sharedFeatures().setWebAnimationsCSSIntegrationEnabled( false );
		RuntimeEnabledFeatures::sharedFeatures().setUserTimingEnabled( false );
		RuntimeEnabledFeatures::sharedFeatures().setResourceTimingEnabled( false );
		RuntimeEnabledFeatures::sharedFeatures().setLinkPreloadEnabled( false );
		RuntimeEnabledFeatures::sharedFeatures().setFetchAPIKeepAliveEnabled( false );
		RuntimeEnabledFeatures::sharedFeatures().setMediaPreloadingEnabled( false );
		RuntimeEnabledFeatures::sharedFeatures().setIsSecureContextAttributeEnabled( false );
		RuntimeEnabledFeatures::sharedFeatures().setDataTransferItemsEnabled( false );;
		RuntimeEnabledFeatures::sharedFeatures().setInspectorAdditionsEnabled( false );
		RuntimeEnabledFeatures::sharedFeatures().setWebSQLDisabled( false );

		settings->setVisualViewportAPIEnabled( false );
		settings->setCSSOMViewScrollingAPIEnabled( false );

		WebFrameNetworkingContext::destroyPrivateBrowsingSession();

		d->mPage->setSessionID( PAL::SessionID::defaultSessionID() );

		settings->setLoadsImagesAutomatically( true );
		settings->setUserStyleSheetLocation( URL() );

		settings->setShouldPrintBackgrounds( false );
		settings->setTextAreasAreResizable( false );
		settings->setEditableLinkBehavior( EditableLinkBehavior::EditableLinkDefaultBehavior );
		settings->setUsesBackForwardCache( true );
		settings->setDOMPasteAllowed( false );
		settings->setShowsURLsInToolTips( false );

		settings->setForceFTPDirectoryListings( true );
		settings->setDeveloperExtrasEnabled( false );
		settings->setNeedsSiteSpecificQuirks( false );

		settings->setFontRenderingMode( FontRenderingMode::Normal );

		settings->setAuthorAndUserStylesEnabled( true );
		settings->setOfflineWebApplicationCacheEnabled( false );

		DatabaseManager::singleton().setIsAvailable( true );
		settings->setLocalStorageEnabled( true );

		settings->setExperimentalNotificationsEnabled( false );

		settings->setWebSecurityEnabled( true );
		settings->setAllowUniversalAccessFromFileURLs( false );
		settings->setJavaScriptCanAccessClipboard( false );
		settings->setXSSAuditorEnabled( true );

#if PLATFORM(WIN)
		DeprecatedGlobalSettings::setShouldUseHighResolutionTimers( true );
#endif

		settings->setFrameFlattening( FrameFlattening::Disabled );
		settings->setShowDebugBorders( false );
		settings->setShowRepaintCounter( false );
		settings->setShowTiledScrollingIndicator( false );

#if ENABLE(WEB_AUDIO)
		settings.setWebAudioEnabled( true );
#endif

#if ENABLE(WEBGL)
		settings->setWebGLEnabled( true );
#endif

		settings->setSpatialNavigationEnabled( false );
		settings->setDNSPrefetchingEnabled( false );
		settings->setHyperlinkAuditingEnabled( true );
		settings->setLoadsSiteIconsIgnoringImageLoadingSetting( false );
		settings->setShowsToolTipOverTruncatedText( false );

#if ENABLE(FULLSCREEN_API)
		settings->setFullScreenEnabled( false );
#endif

		settings->setVideoPlaybackRequiresUserGesture( false );
		settings->setAudioPlaybackRequiresUserGesture( false );

		settings->setAllowsInlineMediaPlayback( true );
		settings->setRequestAnimationFrameEnabled( true );
		settings->setRenderingUpdateThrottlingEnabled( true );

		DeprecatedGlobalSettings::setMockScrollbarsEnabled( false );
		settings->setEnableInheritURIQueryComponent( false );

		settings->setAllowDisplayOfInsecureContent( true );
		settings->setAllowRunningOfInsecureContent( true );

		settings->setJavaScriptRuntimeFlags( JSC::RuntimeFlags( 0 ) );

		RuntimeEnabledFeatures::sharedFeatures().setServerTimingEnabled( false );
		settings->setResizeObserverEnabled( false );
		settings->setCoreMathMLEnabled( false );

		settings->setRequestIdleCallbackEnabled( false );

		settings->setAsyncClipboardAPIEnabled( false );
		settings->setAspectRatioOfImgFromWidthAndHeightEnabled( false );
		settings->setAcceleratedCompositingEnabled( false );
	}

	bool WebView::InitView( const ViewParameters& param )
	{
		TELESCOPE_MEMORY_SCOPE("WebView::InitView");
		
		if ( !d->mInitialized )
		{
			static bool didOneTimeInitialization;
			if ( !didOneTimeInitialization )
			{
				WebPlatformStrategies::initialize();
				auto& memoryPressureHandler = MemoryPressureHandler::singleton();
				memoryPressureHandler.setLowMemoryHandler( []( Critical critical, Synchronous synchronous ) {
					WebCore::releaseMemory( critical, synchronous );
				} );
			}

			d->mWidth = param.mWidth;
			d->mHeight = param.mHeight;
			d->mWindowRect = IntRect( 0, 0, d->mWidth, d->mHeight );

#if USE(TEXTURE_MAPPER_TELESCOPE)
			if ( param.mHardwareRender )
			{
				d->mRender = param.mHardwareRender;
				d->mCompostingContext = new AcceleratedCompositingContext( *this, *d->mRender );
				d->mDisplaySurface = nullptr;
				SetAcceleratedCompositing( true );
			}
			else
#endif // #if USE(TEXTURE_MAPPER_TELESCOPE)
			{
				Telescope::Surface* surface = new Surface( SurfacePixelFormat::PixelFormatARGB32 );
				surface->setSize( param.mWidth, param.mHeight );
				d->mDisplaySurface = surface;
			}

			auto storageProvider = PageStorageSessionProvider::create();

			WebCore::PageConfiguration configuration(
				PAL::SessionID::defaultSessionID(),
				makeUniqueRef<WebEditorClient>( this ),
				WebCore::SocketProvider::create(),
				LibWebRTCProvider::create(),
				CacheStorageProvider::create(),
				BackForwardList::create(),
				CookieJar::create( ( storageProvider ).copyRef() ),
				makeUniqueRef<WebProgressTrackerClient>(),
				makeUniqueRef<MediaRecorderProvider>() );
			//	don't delete, i'm not sure if we can set nullptr or have to new for ps
	//#ifdef SONY_PLATFORM
	//		configuration.chromeClient = nullptr;
	//#else
			configuration.chromeClient = new WebChromeClient( this );
			//#endif
			configuration.inspectorClient = new WebInspectorClient( this );
			configuration.loaderClientForMainFrame = new WebFrameLoaderClient( nullptr, param.mSystemCallback );
			configuration.applicationCacheStorage = &StorageInstance();
			configuration.databaseProvider = &WebDatabaseProvider::singleton();
			configuration.storageNamespaceProvider = &d->mWebViewGroup->storageNamespaceProvider();
			configuration.userContentProvider = &d->mWebViewGroup->userContentController();
			configuration.visitedLinkStore = &d->mWebViewGroup->visitedLinkStore();
			configuration.pluginInfoProvider = &WebPluginInfoProvider::singleton();

			d->mPage = new Page( WTFMove( configuration ) );
			ApplyTelescopeDefaultSettings();

#if USE(TEXTURE_MAPPER_TELESCOPE)
			d->mPage->settings().setAcceleratedCompositingEnabled( isAcceleratedCompositing() );
			d->mPage->settings().setForceCompositingMode( isAcceleratedCompositing() );
#endif // #if USE(TEXTURE_MAPPER_TELESCOPE)

			storageProvider->setPage( *d->mPage );

			d->mPage->addLayoutMilestones( { DidFirstLayout, DidFirstVisuallyNonEmptyLayout } );

			WebFrame* webFrame = WebFrame::createInstance();
			webFrame->initWithWebView( this, d->mPage );
			static_cast< WebFrameLoaderClient& >( d->mPage->mainFrame().loader().client() ).setWebFrame( webFrame );
			static_cast< WebProgressTrackerClient& >( d->mPage->progress().client() ).setWebFrame( webFrame );
			d->mMainFrame = webFrame;
			d->mPage->mainFrame().tree().setName( "" );
			d->mPage->mainFrame().init();
		}

		d->mInitialized = true;
		return true;
	}

	void WebView::ShutdownView( void )
	{
		if ( d->mPage )
		{
			d->mPage->mainFrame().loader().detachFromParent();
		}

		d->mWebViewGroup->removeWebView( this );
		if ( d->mDisplaySurface )
		{
			d->mDisplaySurface->reset();
			delete d->mDisplaySurface;
		}
#if USE(TEXTURE_MAPPER_TELESCOPE)
		if ( d->mCompostingContext )
		{
			delete d->mCompostingContext;
			d->mCompostingContext = nullptr;
		}
#endif // #if USE(TEXTURE_MAPPER_TELESCOPE)
		SetAcceleratedCompositing( false );
	}

	static void fastClearBackingBufferSurface( const SurfaceDescriptor* desc )
	{
		const int stride = desc->stride;
		const int bpp = bytesPerPixel( desc->pixelFormat );
		char* data = (char*)desc->data;
		const bool isBufferConsecutive = desc->stride == desc->width * bpp;

		if ( isBufferConsecutive )
		{
			memset( data, 0, desc->height * desc->stride );
		}
		else
		{
			const int rowBufferSize = desc->width * bpp;
			for ( int i = 0; i < desc->height; i++ )
			{
				memset( data + i * stride, 0,  rowBufferSize );
			}
		}
	}

	void WebView::PaintRegion( const WebCore::IntRect* dirtyRegion )
	{
		Frame* coreFrame = core( d->mMainFrame );
		ASSERT( coreFrame );

		FrameView* frameView = coreFrame->view();

		SurfaceDescriptor descriptor;
		SurfaceRegion region( dirtyRegion->x(), dirtyRegion->y(), dirtyRegion->width(), dirtyRegion->height() );
		d->mDisplaySurface->getData( descriptor, &region );

		// clear the dirty region
		fastClearBackingBufferSurface( &descriptor );

		RefPtr<cairo_surface_t> cairoSurface = adoptRef( cairo_image_surface_create_for_data( static_cast< unsigned char* >( descriptor.data ), CAIRO_FORMAT_ARGB32, descriptor.width, descriptor.height, descriptor.stride ) );
		RefPtr<cairo_t> cairoContext = adoptRef( cairo_create( ( cairoSurface ).get() ) );
		PlatformContextCairo platformContextCairo( cairoContext.get() );
		WebCore::GraphicsContext graphicsContext( &platformContextCairo );

		if ( !graphicsContext.paintingDisabled() )
		{
			// redraw the dirty region
			graphicsContext.translate( -dirtyRegion->x(), -dirtyRegion->y() );
			frameView->paint( graphicsContext, *dirtyRegion );
		}
		ASSERT( cairo_status( ( cairoContext ).get() ) == CAIRO_STATUS_SUCCESS );
	}

	static void PrintDirtyRegions( DirtyRegions* dirtyRegions )
	{
		if( !dirtyRegions->Empty() )
		{
			char logBuffer[1024] = "";
			for( unsigned i = 0; i < dirtyRegions->Count(); i++ )
			{
				auto region = dirtyRegions->GetRegion( i );
				char logBufferRegion[128] = "";
				snprintf( logBufferRegion, sizeof(logBufferRegion), "{%d, %d, %d, %d}, ", region->x(), region->y(), region->x() + region->width(), region->y() + region->height() );
				if ( strlen(logBuffer) + strlen(logBufferRegion) > sizeof(logBuffer) - 1 )
					break;
				strcat( logBuffer, logBufferRegion );
			}
			int length = strlen(logBuffer);
			if ( length >= 2 )
				logBuffer[length - 2] = '\0';
			Telescope::LogDebug( Telescope::LogChannel::GENERAL, "%d dirty regions detected: %s", dirtyRegions->Count(), logBuffer );
		}
	}

	void WebView::SetAcceleratedCompositing( bool accelerated )
	{
#if USE(TEXTURE_MAPPER_TELESCOPE)
		if( d->mIsAcceleratedCompositing == accelerated )
			return;
		d->mIsAcceleratedCompositing = accelerated;
#endif // #if USE(TEXTURE_MAPPER_TELESCOPE)
	}

	void WebView::Paint()
	{
		TELESCOPE_MEMORY_SCOPE("WebView::Paint");

		Page* page = d->mPage;
		page->updateRendering();

		Frame* coreFrame = core( d->mMainFrame );
		if ( !coreFrame )
		{
			return;
		}

#if USE(TEXTURE_MAPPER_TELESCOPE)
		if ( isAcceleratedCompositing() )
		{
			d->mCompostingContext->flushAndRenderLayers();
		}
		else
#endif // #if USE(TEXTURE_MAPPER_TELESCOPE)
		{
			PrintDirtyRegions( &d->mDirtyRegions );
			if ( !d->mDirtyRegions.Empty() )
			{
				if ( d->mDirtyRegions.IsOverflow() )
				{
					PaintRegion( windowRect() );
				}
				else
				{
					for ( unsigned i = 0; i < d->mDirtyRegions.Count(); i++ )
					{
						PaintRegion( d->mDirtyRegions.GetRegion( i ) );
					}
				}
				d->mDirtyRegions.Clear();
			}
		}
	}

	void WebView::SetURI( const wchar_t* url )
	{
		if ( url && url[0] )
		{
#ifdef SONY_PLATFORM
#else
			WTF::URL kurl( WTF::URL(), url );
			d->mMainFrame->loadRequest( &kurl );
#endif
		}
	}

	void WebView::SetURI( const char* url )
	{
		if ( url && url[0] )
		{
			WTF::URL kurl( WTF::URL(), url );
			d->mMainFrame->loadRequest( &kurl );
		}
	}

	void WebView::GetBuffer( void*& buffer, int& bufferSize )
	{
#if USE(TEXTURE_MAPPER_TELESCOPE)
		if ( isAcceleratedCompositing() )
		{
			buffer = nullptr;
			bufferSize = 0;
		}
		else
#endif // #if USE(TEXTURE_MAPPER_TELESCOPE)
		{
			SurfaceDescriptor surfaceDesc;
			d->mDisplaySurface->getData( surfaceDesc );
			buffer = surfaceDesc.data;
			bufferSize = surfaceDesc.stride * surfaceDesc.height;
		}
	}

	void WebView::SetSize( int width, int height )
	{
		TELESCOPE_MEMORY_SCOPE("WebView::SetSize");

		d->mWidth = width;
		d->mHeight = height;
		d->mWindowRect = IntRect( 0, 0, width, height );
		d->mDirtyRegions.Clear();

#if USE(TEXTURE_MAPPER_TELESCOPE)
		if ( isAcceleratedCompositing() )
		{
			d->mCompostingContext->resizeRootLayer( IntSize( width, height ) );
		}
		else
#endif // #if USE(TEXTURE_MAPPER_TELESCOPE)
		{
			assert( d->mDisplaySurface );
			d->mDisplaySurface->setSize( width, height );
		}

		if ( d->mMainFrame &&                    // Telescope::WebFrame
			d->mMainFrame->frameView() &&        // WebCore::Frame
			d->mMainFrame->frameView()->view() ) // WebCore::FrameView
		{
			WebCore::FrameView* nativeFrameView = d->mMainFrame->frameView()->view();
			nativeFrameView->resize( width, height );
		}
	}

	int WebView::Width() const
	{
		return d->mWidth;
	}

	int WebView::Height() const
	{
		return d->mHeight;
	}

	WebCore::IntRect* WebView::windowRect()
	{
		return &d->mWindowRect;
	}

	bool WebView::canHandleRequest( const WebCore::ResourceRequest* request )
	{
		return true;
	}

	bool WebView::transparent() const
	{
		return d->mTransparent;
	}

	void WebView::setTransparent( bool transparent )
	{
		d->mTransparent = transparent;
	}

	Telescope::IOperation* WebView::GetOperationInterface()
	{
		return Telescope::Telescope_GetOperation();
	}

	Frame* WebView::TopLevelFrame()
	{
		return core( d->mMainFrame );
	}

	void WebView::Refresh( bool ignoreCache )
	{
		WTF::OptionSet option( ignoreCache ? WebCore::ReloadOption::FromOrigin : WebCore::ReloadOption::ExpiredOnly );
		d->mMainFrame->frameView()->loader().reload( option );
	}

	void WebView::BindJavaScriptObject( const char* name, Telescope::IJSBoundObject* obj )
	{
		if ( d->mPage && d->mMainFrame )
		{
			d->mBoundObjects[name] = obj;
			d->mMainFrame->AddToJavaScriptWindowObject( name, obj );
		}
	}

	void WebView::RebindJavaScriptObject()
	{
		for ( auto it = d->mBoundObjects.cbegin(); it != d->mBoundObjects.cend(); ++it )
		{
			d->mMainFrame->AddToJavaScriptWindowObject( it->first, it->second );
		}
	}

	TelescopeJSValue* WebView::CreateJavaScriptValue()
	{
		return d->mMainFrame->CreateJavaScriptValue();
	}

	void WebView::DestroyJavascriptValue( TelescopeJSValue* v )
	{
		d->mMainFrame->DestroyJavaScriptValue( v );
	}


#if TELESCOPE_USING(TELESCOPE_PC_PROGRAM)
	void WebView::SetHWND( void* hWnd )
	{
		ime->SetViewWnd( ( HWND )hWnd );
	}

	void WebView::OnIMEMsg( unsigned int message, unsigned long long wParam, long long lParam )
	{
		char caBuf[256];
		sprintf_s( caBuf, "[%I64u] MSG: %d, wParam: %I64u, lParam: %I64d\n", GetTickCount64(), message, wParam, lParam );
		OutputDebugStringA( caBuf );

		bool handled = true;
		LRESULT lResult = 0;
		switch ( message )
		{
		case WM_IME_STARTCOMPOSITION:
			handled = ime->onIMEStartComposition( this );
			break;
		case WM_IME_REQUEST:
			lResult = ime->onIMERequest( this, wParam, lParam );
			break;
		case WM_IME_COMPOSITION:
			handled = ime->onIMEComposition( this, lParam );
			break;
		case WM_IME_ENDCOMPOSITION:
			handled = ime->onIMEEndComposition( this );
			break;
		case WM_IME_CHAR:
			handled = ime->onIMEChar( this, wParam, lParam );
			break;
		case WM_IME_NOTIFY:
			LRESULT lResult;
			handled = ime->onIMENotify( this, wParam, lParam, &lResult );
			break;
		case WM_IME_SELECT:
			handled = ime->onIMESelect( this, wParam, lParam );
			break;
		case WM_IME_SETCONTEXT:
			handled = ime->onIMESetContext( this, wParam, lParam );
			break;
		default:
			break;
		}
	}
#else
	void WebView::OnIMEMsg( unsigned int message, unsigned long long wParam, long long lParam )
	{

	}

	void WebView::SetHWND( void* hWnd )
	{

	}
#endif

	void WebView::AddDirtyRegion( const WebCore::IntRect* region )
	{

		if ( isAcceleratedCompositing() )
		{
#if USE(TEXTURE_MAPPER_TELESCOPE)
			d->mCompostingContext->setNonCompositedContentsNeedDisplay( *region );
#endif // #if USE(TEXTURE_MAPPER_TELESCOPE)
			return;
		}

		auto validRegion = *windowRect();
		validRegion.intersect( *region );
		if ( validRegion.width() != 0 && validRegion.height() != 0 )
		{
			d->mDirtyRegions.AddRegion( &validRegion );
		}
	}

	bool WebView::isAcceleratedCompositing() const
	{
#if USE(TEXTURE_MAPPER_TELESCOPE)
		return d->mIsAcceleratedCompositing;
#else
		return false;
#endif	// #if USE(TEXTURE_MAPPER_TELESCOPE)
	}

	const HardwareSurfaceBase* WebView::GetViewHardwareSurface() const
	{
#if USE(TEXTURE_MAPPER_TELESCOPE)
		assert(isAcceleratedCompositing());
		if (isAcceleratedCompositing())
		{
			return d->mCompostingContext->getRenderSurface();
		}
#endif	// #if USE(TEXTURE_MAPPER_TELESCOPE)
		return nullptr;
	}

	void WebView::flushPendingGraphicsLayerChangesSoon()
	{
#if USE(TEXTURE_MAPPER_TELESCOPE)
		if (d->mCompostingContext)
		{
			d->mCompostingContext->flushPendingLayerChangesSoon();
		}
#endif	// #if USE(TEXTURE_MAPPER_TELESCOPE)
	}

	void WebView::SetRootChildLayer(GraphicsLayer* layer)
	{
#if USE(TEXTURE_MAPPER_TELESCOPE)
		if (!d->mCompostingContext)
		{
			return;
		}
		SetAcceleratedCompositing(layer ? true : false);
		d->mCompostingContext->setRootCompositingLayer(layer);
#endif	// #if USE(TEXTURE_MAPPER_TELESCOPE)
	}
}
