/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#pragma once

#include <memory.h>
#include <ClientWrappers/MemoryAllocatorBase.h>

namespace Telescope
{
	constexpr const char* HTTP_MEMORY_TAG = "http";

	constexpr const size_t METHOD_LEN = 8;
	constexpr const size_t MAXURL_LEN = 2 * 1024;
	constexpr const size_t COMMON_BUFFER_SIZE = 8 * 1024;

	constexpr const unsigned int CTX_FLAG_PAUSE          = ( 1 << 0 );
	constexpr const unsigned int CTX_FLAG_PAUSE_CHANGE   = ( 1 << 1 );
	constexpr const unsigned int CTX_FLAG_CHUNK_ENCODING = ( 1 << 2 );
	constexpr const unsigned int CTX_FLAG_DISABLE_SSL    = ( 1 << 3 );
	constexpr const unsigned int CTX_FLAG_IS_HTTPS       = ( 1 << 4 );

	// Must match CTX_READFUNC_ABORT CTX_WRITEFUNC_PAUSE in CurlReplacement.h
	constexpr const unsigned int HTTP_READFUNC_ABORT  = 0x10000000;
	constexpr const unsigned int HTTP_WRITEFUNC_PAUSE = 0x10000001;

	// Must match HTTP_VERSION_1_0 HTTP_VERSION_1_1 HTTP_VERSION_2_0
	enum
	{
		HTTP_VERSION_1_0 = 1,
		HTTP_VERSION_1_1 = 2,
		HTTP_VERSION_2_0 = 3,
	};

	// Must match CurlReplacement.h CRECode
	enum
	{
		HTTP_WRAPPER_OK,
		HTTP_WRAPPER_FAILED_INIT,
		HTTP_WRAPPER_AGAIN,
		HTTP_WRAPPER_RECV_ERROR,
		HTTP_WRAPPER_TOO_MANY_REDIRECTS,
		HTTP_WRAPPER_OPERATION_TIMEDOUT,
		HTTP_WRAPPER_OUT_OF_MEMORY,
		HTTP_WRAPPER_SSL_CACERT,
		HTTP_WRAPPER_SSL_CONNECT_ERROR,
		HTTP_WRAPPER_PEER_FAILED_VERIFICATION,
		HTTP_WRAPPER_ADD_HEADER_ERROR,
		HTTP_WRAPPER_MANIFEST_VALIDATION,
		HTTP_WRAPPER_ADD_HEADER_BUFFER_TOO_SMALL,
		HTTP_WRAPPER_REMOVE_HEADER_BUFFER_TOO_SMALL,
		HTTP_WRAPPER_CALLBACK_BUFFER_TOO_SMALL,
		HTTP_WRAPPER_XR012_ERROR,
		HTTP_WRAPPER_LAST,
	};

	enum
	{
		HTTP_REQUEST_STATUS_WAITING,
		HTTP_REQUEST_STATUS_WORKING,
		HTTP_REQUEST_STATUS_DONE,
		HTTP_REQUEST_STATUS_FAIL,
		HTTP_REQUEST_STATUS_ABORTED,
	};

	typedef size_t( *SendRecvCallback )( char* ptr, size_t blockSize, size_t numberOfBlocks, void* userData );

	struct HttpIdentify
	{
		struct
		{
			int connectionId;
			int requestId;
		} sony;
		struct
		{
			void* m_hSession;
			void* volatile m_hConnect;
			void* volatile m_hRequest;
		} ms;
	};

	struct SendRecvCallbackConfig
	{
		SendRecvCallback callback;
		void* data;

		SendRecvCallbackConfig()
			: callback( nullptr )
			, data( nullptr )
		{}
	};

	struct IManifestFilter
	{
		virtual void AddData(const void* pData, size_t bytes) = 0;
		virtual bool Verify() = 0;
		virtual void Release() = 0;
	};

	struct HttpContext
	{
		// identify
		HttpIdentify identify;
		// configuration
		SendRecvCallbackConfig headerRecvCB;
		SendRecvCallbackConfig dataRecvCB;
		SendRecvCallbackConfig dataSendCB;

		// data transmission
		const void* simpleUploadBuffer;
		void* commonBuffer;

		unsigned long long timeout;
		int headerDownload;
		int currentUpload;
		int totalUpload;
		int pendingDownload;
		int currentDownload;
		int totalDownload;
		int simpleUploadSize;

		// status data
		int flags;
		int status;
		int internalStatus;
		int lastErrorCode;
		int certStatus;
		int httpVersion;
		unsigned int httpStatusCode;

		char method[METHOD_LEN];
		char url[MAXURL_LEN];

		IMemoryAllocator* memoryAllocator;

		// request filter
		IManifestFilter* manifestFilter;

		unsigned long long processTimeCounterStamp;

		HttpContext(IMemoryAllocator* pMemoryAllocator)
			: simpleUploadBuffer( nullptr )
			, timeout( 0 )
			, headerDownload( 0 )
			, currentUpload( 0 )
			, totalUpload( 0 )
			, pendingDownload( 0 )
			, currentDownload( 0 )
			, totalDownload( 0 )
			, simpleUploadSize( 0 )
			, flags( 0 )
			, status( 0 )
			, internalStatus( 0 )
			, lastErrorCode( 0 )
			, certStatus( 0 )
			, httpVersion( 0 )
			, httpStatusCode( 0 )
			, memoryAllocator(pMemoryAllocator)
			, manifestFilter(nullptr)
			, processTimeCounterStamp(0)
		{
			memset( &identify, 0, sizeof( identify ) );
			memset( method, 0, METHOD_LEN );
			memset( url, 0, MAXURL_LEN );

			if( memoryAllocator )
				commonBuffer = memoryAllocator->Allocate( COMMON_BUFFER_SIZE, HTTP_MEMORY_TAG );
			else
				commonBuffer = nullptr;
		}

		~HttpContext()
		{
			if( memoryAllocator && commonBuffer )
				memoryAllocator->Deallocate( commonBuffer );

			if (manifestFilter)
				manifestFilter->Release();
		}
	};

	class HttpWrapperBase
	{
	public:
		virtual bool        initRequest( HttpContext* context, const char* method, const char* url, const unsigned int timeout ) = 0;
		virtual void        addRequestHeader( HttpContext* context, const char* name, const char* value ) = 0;
		virtual void        removeRequestHeader( HttpContext* context, const char* name ) = 0;
		virtual void        perform( HttpContext* context ) = 0;
		virtual void        abort( HttpContext* context ) = 0;
		virtual void        finialize( HttpContext* context ) = 0;
		virtual long        getHttpVersion( HttpContext* context ) = 0;
		virtual int         getHttpStatus( HttpContext* context ) = 0;

		// These following utility interfaces should be implemented in HttpWrapperBaseImpl within TelescopeClient
		virtual void setPause( HttpContext* context, bool doPause ) = 0;
		virtual bool isPause( HttpContext* context ) = 0;
		virtual bool isPauseChange( HttpContext* context ) = 0;
		virtual void setSimpleUploaderBuffer( HttpContext* context, const void* buffer, long size) = 0;
		virtual void setContentLength( HttpContext* context, int contentLength ) = 0;
		virtual int  getContentLength( HttpContext* context ) = 0;
		virtual void setHeaderReceiveCallback( HttpContext* context, SendRecvCallback callback, void* userData ) = 0;
		virtual void setDataSendCallback( HttpContext* context, SendRecvCallback callback, void* userData ) = 0;
		virtual void setDataReceiveCallback( HttpContext* context, SendRecvCallback callback, void* userData ) = 0;
		virtual void setChunkedTransferEnabled( HttpContext* context, bool enabled ) = 0;
		virtual bool isChunkedTransferEnabled( HttpContext* context ) = 0;
		virtual void disableServerTrustEvaluation( HttpContext* context ) = 0;
		virtual bool isDisableServerTrustEvaluation( HttpContext* context ) = 0;
		virtual int  getStatus( HttpContext* context ) = 0;
		virtual int  getLastErrorCode( HttpContext* context ) = 0;

		virtual ~HttpWrapperBase() {}
	};
}
