/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include <string.h>

namespace Telescope::JsonWrapper
{
	struct IValue;
	struct IObject;
	struct IArray;
	struct IString;

	//
	// IRelease
	//

	struct IRelease
	{
		// We provide the virtual destructor just for letting Sony PlayStation compilers shut up.
		// We should NOT call it indirectly by an 'operator delete'.
		// Because IRelease objects can have a different memory allocator than what you are using.
		// Use the method Release() to do what 'operator delete' should have done.
		virtual ~IRelease()	{}

		virtual void Release() = 0;
	};


	//
	// InterfacePtr
	//

	template<typename TRelease>
	class InterfacePtr
	{
		TRelease* m_pRelease;

	public:
		InterfacePtr()
			: m_pRelease(nullptr)
		{}

		InterfacePtr(TRelease* pRelease)
			: m_pRelease(pRelease)
		{}

		InterfacePtr(InterfacePtr&& other)
			: m_pRelease(other.Release())
		{}

		~InterfacePtr()
		{
			if (m_pRelease)
				m_pRelease->Release();
		}

		InterfacePtr(const InterfacePtr&) = delete;;
		InterfacePtr& operator =(const InterfacePtr&) = delete;;

		void Reset(TRelease* pRelease = nullptr)
		{
			if (m_pRelease)
				m_pRelease->Release();

			m_pRelease = pRelease;
		}

		TRelease* Release()
		{
			auto p = m_pRelease;

			m_pRelease = nullptr;

			return p;
		}

		TRelease* operator ->() const
		{
			return m_pRelease;
		}

		TRelease* Get() const
		{
			return m_pRelease;
		}
	};

	typedef InterfacePtr<IValue> ValuePtr;
	typedef InterfacePtr<IObject> ObjectPtr;
	typedef InterfacePtr<IArray> ArrayPtr;
	typedef InterfacePtr<IString> StringPtr;


	//
	// IString
	//

	struct IString : public IRelease
	{
		virtual const char* CString() const = 0;
		
		operator const char* () const { return CString(); }
		size_t Length() const { return strlen(CString()); }
	};


	//
	// IValue
	//

	struct IValue : public IRelease
	{
		virtual bool GetBoolean(bool& v) const = 0;
		virtual bool GetInteger(int& v) const = 0;
		virtual bool GetInteger(unsigned& v) const = 0;
		virtual bool GetInteger(long& v) const = 0;
		virtual bool GetInteger(long long& v) const = 0;
		virtual bool GetInteger(unsigned long& v) const = 0;
		virtual bool GetInteger(unsigned long long& v) const = 0;
		virtual bool GetDouble(double& v) const = 0;
		virtual bool GetFloat(float& v) const = 0;
		virtual bool GetString(IString*& v) const = 0;
		virtual StringPtr GetString() const = 0;

		virtual ObjectPtr AsObject() const = 0;
		virtual ArrayPtr AsArray() const = 0;
	};


	//
	// IObject
	//

	struct IObject : public IRelease
	{
		struct IIterator : public IRelease
		{
			virtual bool End() const = 0;
			virtual void Next() = 0;
			virtual const char* Name() const = 0;
			virtual ValuePtr Value() const = 0;
		};

		typedef InterfacePtr<IIterator> IteratorPtr;

		virtual ValuePtr GetValue(const char* pName) const = 0;
		virtual ObjectPtr GetObject(const char* pName) const = 0;
		virtual ArrayPtr GetArray(const char* pName) const = 0;

		virtual IteratorPtr GetIterator() const = 0;
	};


	//
	// IArray
	//

	struct IArray : public IRelease
	{
		virtual size_t Length() const = 0;
		virtual ValuePtr GetElement(size_t index) const = 0;
	};

}