/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#pragma once

namespace JSC
{
	class JSGlobalObject;
	class JSValue;
}

namespace Telescope
{
	class JSValuePropertyIterator;

	enum EJSValueType
	{
		EJVT_Null,
		EJVT_Undefined,
		EJVT_Number,
		EJVT_Boolean,
		EJVT_StringType,
		EJVT_Array,
		EJVT_Object,
		EJVT_Function,
		EJVT_Unknown
	};

	class TelescopeJSValue
	{
		friend class JSValuePropertyIterator;

	public:
		TelescopeJSValue();
		TelescopeJSValue( const TelescopeJSValue& rSrc );
		TelescopeJSValue( JSC::JSValue* pVal, JSC::JSGlobalObject* pGlobalObject );

		virtual ~TelescopeJSValue();

		void operator=( const TelescopeJSValue& rSrc );

		virtual EJSValueType Type() const;

		virtual void SetNull();
		virtual void SetUndefined();

		virtual void SetNumberValue( double dblVal );
		virtual void SetNumberValue( int iVal );
		virtual double GetNumberValue() const;

		virtual bool IsInt32() const;
		virtual int GetInt32() const;

		virtual bool IsDouble() const;
		virtual double GetDouble() const;

		virtual void SetBooleanValue( bool bVal );
		virtual bool GetBooleanValue() const;

		virtual void SetStringValue( const void* pVal, bool isWideChar );
		virtual bool GetStringValue( void* pInBuf, unsigned int& inoutBufSize, bool& outWideChar ) const;

		virtual void SetArraySize( size_t stSize = 0 );
		virtual void SetArrayValue( size_t stIndex, const TelescopeJSValue& rVal );
		virtual void SetArrayValues( const TelescopeJSValue* pVal, size_t stCnt, size_t stStartIndex = 0 );
		virtual void PushArrayValue( const TelescopeJSValue& rVal );
		virtual void PopArrayValue( TelescopeJSValue* pValOut );
		virtual void GetArrayValue( size_t stIndex, TelescopeJSValue* pValOut ) const;
		virtual size_t GetArrayLength() const;

		virtual void SetObjectType();
		virtual void SetProperty( const char* pKey, const TelescopeJSValue& rVal );
		virtual void GetProperty( const char* pKey, TelescopeJSValue* pValOut ) const;

		virtual JSValuePropertyIterator* First();
		virtual JSValuePropertyIterator* GetNext();

		virtual bool Call( TelescopeJSValue* pArgs, size_t stCnt, TelescopeJSValue* pValOut = nullptr );

		void Assign( const JSC::JSValue* pVal );
		void SetGlobalObject( JSC::JSGlobalObject* pGlobalObject );
		JSC::JSGlobalObject* GetGlobalObject();
		JSC::JSValue* GetImpl() const;

	private:
		virtual void Destruct();
		virtual void Copy( const TelescopeJSValue& rSrc );

		static const int sciJSValueSize = 8;
		char m_caJSValue[sciJSValueSize];
		JSC::JSGlobalObject* m_pGlobalObject;
		JSValuePropertyIterator* m_pIterator;
	};
}
