/*
 * Copyright (C) 2020 Activision Publishing, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. Neither the name of Activision Publishing, Inc. nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef __TELESCOPE_OPERATION_H__
#define __TELESCOPE_OPERATION_H__


namespace WebCore
{
	class PlatformMouseEvent;
	class PlatformWheelEvent;
	class Frame;
}

namespace Telescope
{
	//////////////////////////////////////////////////////////////////////////
	//	Delete these comments when publish
	//////////////////////////////////////////////////////////////////////////

	const unsigned int g_uiKModifierMaskNone = 0x0000;
	const unsigned int g_uiKModifierMaskAny = 0x03ff;
	const unsigned int g_uiKModifierMaskShift = 0x0001;
	const unsigned int g_uiKModifierMaskControl = 0x0002;
	const unsigned int g_uiKModifierMaskAlt = 0x0004;
	const unsigned int g_uiKModifierMaskOption = 0x0008;
	const unsigned int g_uiKModifierMaskOS = 0x0010;
	const unsigned int g_uiKModifierMaskCapsLock = 0x0020;
	const unsigned int g_uiKModifierMaskNumLock = 0x0040;
	const unsigned int g_uiKModifierMaskScrollLock = 0x0080;
	const unsigned int g_uiKModifierMaskCompose = 0x0100;
	const unsigned int g_uiKModifierMaskKana = 0x0200;

	const int PC_WHEEL_UP_INDEX = 21;
	const int PC_WHEEL_DOWN_INDEX = 22;
	const int MAX_JS_CMD_LEN = 256;

	const unsigned int g_uiKMouseLeft = 0;
	const unsigned int g_uiKMouseMiddle = 1;
	const unsigned int g_uiKMouseRight = 2;
	const unsigned int g_uiKMouseX1 = 3;
	const unsigned int g_uiKMouseX2 = 4;
	const unsigned int g_uiKMouseNoButton = 0xffffffff;

	struct MouseMoveEvent
	{
		unsigned int   m_uID;
		int			   m_iX;
		int			   m_iY;
		int			   m_iGlobalX;
		int            m_iGlobalY;
		unsigned int   m_uModifiers;

		MouseMoveEvent(
			unsigned int id = g_uiKMouseNoButton,
			int			 x = -1,
			int			 y = -1,
			int			 globalX = -1,
			int			 globalY = -1,
			unsigned int modifiers = 0 )
			: m_uID( id ),
			m_iX( x ),
			m_iY( y ),
			m_iGlobalX( globalX ),
			m_iGlobalY( globalY ),
			m_uModifiers( modifiers )
		{
		}
	};

	struct MouseButtonEvent
	{
		unsigned int  m_uID;
		int           m_iX;
		int           m_iY;
		int           m_iGlobalX;
		int           m_iGlobalY;
		unsigned int  m_uModifiers;
		unsigned int  m_uClickCount;
		bool          m_bDepressed;

		MouseButtonEvent(
			unsigned int id = g_uiKMouseNoButton,
			bool bDepressed = false,
			unsigned int clickCount = 1,
			int x = -1,
			int y = -1,
			int globalX = -1,
			int globalY = -1,
			unsigned int modifiers = 0 )
			: m_uID( id ),
			m_iX( x ),
			m_iY( y ),
			m_iGlobalX( globalX ),
			m_iGlobalY( globalY ),
			m_uModifiers( modifiers ),
			m_uClickCount( clickCount ),
			m_bDepressed( bDepressed )
		{
		}
	};

	struct MouseWheelEvent
	{
		int             m_iZDelta;
		float           m_iNumLines;
		int             m_iX;
		int             m_iY;
		int             m_iGlobalX;
		int             m_iGlobalY;
		unsigned int	m_uModifiers;

		MouseWheelEvent(
			int          zDelta = 0,
			float        numLines = 0.f,
			int          x = -1,
			int          y = -1,
			int          globalX = -1,
			int          globalY = -1,
			unsigned int modifiers = 0 )
			: m_iZDelta( zDelta ),
			m_iNumLines( numLines ),
			m_iX( x ),
			m_iY( y ),
			m_iGlobalX( globalX ),
			m_iGlobalY( globalY ),
			m_uModifiers( modifiers )
		{
		}
	};

class IOperation
{
public:
	virtual ~IOperation() {}

	IOperation() {}
	IOperation( const IOperation&  ) = delete;
	IOperation( const IOperation&& ) = delete;
	void operator=  ( const IOperation&  ) = delete;
	void operator=  ( const IOperation&& ) = delete;

private:
	void BuildPlatformMouseMoveEvent( const MouseMoveEvent& rEv, WebCore::PlatformMouseEvent* pOut );
	void BuildPlatformMouseButtonEvent( const MouseButtonEvent& rEv, WebCore::PlatformMouseEvent* pOut );
	void BuildPlatformMouseWheelEvent( const MouseWheelEvent& rEv, WebCore::PlatformWheelEvent* pOut );

public:
	virtual void evaluateJavaScriptAsVoid( const char* scriptSource, WebCore::Frame* pFrame );
	virtual int evaluateJavaScriptAsInt32( const char* scriptSource, WebCore::Frame* pFrame );
	virtual unsigned int evaluateJavaScriptAsUInt32( const char* scriptSource, WebCore::Frame* pFrame );
	virtual long long evaluateJavaScriptAsAnyInt( const char* scriptSource, WebCore::Frame* pFrame );
	virtual double evaluateJavaScriptAsDouble( const char* scriptSource, WebCore::Frame* pFrame );
	virtual bool evaluateJavaScriptAsBoolean( const char* scriptSource, WebCore::Frame* pFrame, bool* pFailed );

	virtual void OnMouseMoveEvent( const MouseMoveEvent& rEv, WebCore::Frame* pFrame );
	virtual void OnMouseButtonEvent( const MouseButtonEvent& rEv, WebCore::Frame* pFrame );
	virtual void OnMouseWheelEvent( const MouseWheelEvent& rEv, WebCore::Frame* pFrame );

	virtual void ProcessKeyboardKey( unsigned long long wParam, long long lParam, bool bKeyDown, WebCore::Frame* pFrame );
	virtual void ProcessKeyboardInputChar( unsigned long long wParam, long long lParam, bool eraseAll, WebCore::Frame* pFrame );
	virtual void ProcessKeyboardInputString( const char* str, bool eraseAll, WebCore::Frame* pFrame );
	
	virtual void SetFocus( bool focused, WebCore::Frame* pFrame );

	virtual void OnGainFocus( WebCore::Frame* pFrame );
	virtual void OnLoseFocus( WebCore::Frame* pFrame );
	virtual bool OnFocusNextElement( WebCore::Frame* pFrame, int iIndex );

	virtual void OnGamePadPressed( WebCore::Frame* pFrame, int iButtonIndex );

	virtual void BindJSObject(WebCore::Frame* pFrame, const char* pObjStringVal);
};

IOperation* Telescope_GetOperation();
}

#endif // #ifndef __TELESCOPE_OPERATION_H__
